!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    spectra.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Spectras
  USE TreeDeclarations
  USE DataDeclarations
  USE PhysicsDeclarations
  USE GlobalDeclarations
  USE EllipticDeclarations
  USE EOS
  USE ParticleDeclarations
  USE Fields
  USE IOBOV
  USE ProcessingDeclarations
  USE PFFT
  USE LayoutControl
  USE LayoutComms
  USE SlopeLim
  IMPLICIT NONE
  INTEGER, PARAMETER :: MAXIMUMSPECTRALEVEL=-1
  INTEGER, PARAMETER :: SCALAR_SPECT=0, VECTOR_SPECT=1
  INTEGER, PARAMETER :: NO_WINDOW=0, COSINE_WINDOW=1
  TYPE SpectraDef
     TYPE(FieldDef), ALLOCATABLE :: Fields(:)
     REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: Data
     INTEGER :: Level=MAXIMUMSPECTRALEVEL
     INTEGER, DIMENSION(3,2) :: mB
     INTEGER :: WindowFunction = NO_WINDOW
     REAL(KIND=qPREC) :: kmin
     REAL(KIND=qPREC) :: dk=1d0
     INTEGER :: Type = SCALAR_SPECT
     INTEGER :: method=CONSTANT_INTERP
     TYPE(SpectraDef), POINTER :: next      
  END TYPE SpectraDef

  TYPE(SpectraDef), POINTER :: FirstSpectra, LastSpectra

CONTAINS
  SUBROUTINE SpectraInit()
    Nullify(FirstSpectra, LastSpectra)
  END SUBROUTINE SpectraInit

  SUBROUTINE CreateSpectra(Spectra, levelOpt)
    TYPE(SpectraDef), POINTER :: Spectra
    INTEGER, OPTIONAL :: levelOpt
    ALLOCATE(Spectra)
    CALL AddSpectraToList(Spectra)
    IF (present(levelOpt)) THEN
       Spectra%level = levelopt
       Spectra%mB(:,1)=1
       Spectra%mB(:,2)=levels(levelopt)%mX
    END IF
    NULLIFY(Spectra%next)
  END SUBROUTINE CreateSpectra

  SUBROUTINE DestroySpectra(Spectra)
    TYPE(SpectraDef), POINTER :: Spectra
    DEALLOCATE(Spectra)
    NULLIFY(Spectra)
  END SUBROUTINE DestroySpectra

  SUBROUTINE AddSpectraToList(Spectra)
    TYPE(SpectraDef),POINTER :: Spectra
    IF(.NOT. ASSOCIATED(FirstSpectra)) THEN ! First Spectra Object only
       FirstSpectra=>Spectra
       LastSpectra=>Spectra
    ELSE
       LastSpectra%next=>Spectra
       LastSpectra=>Spectra
    END IF
  END SUBROUTINE AddSpectraToList

  SUBROUTINE ProcessSpectras
    TYPE(SpectraDef), POINTER :: Spectra
    TYPE(PFFTPlanDef), POINTER :: plan
    INTEGER :: n,l,SpectraLevel, mB(3,2), mBw(3,2)
    TYPE(NodeDefList), POINTER :: NodeList
    TYPE(ParticleListDef), POINTER :: ParticleList
    CHARACTER(LEN=25) :: FileName
    INTEGER :: i,j,k
    REAL(KIND=qPREC) :: x,y,z, origin(3), mx(3), r, weight
    COMPLEX(8), DIMENSION(:,:,:,:), POINTER :: data, tempdata
    TYPE(LayoutDef), POINTER :: spectra_layout
    CHARACTER(LEN=5) :: str
    IF (MPI_ID == 0 .AND. ASSOCIATED(FirstSpectra)) THEN
       WRITE(FileName, '(A12,I5.5,A6)') 'out/Spectra_', current_frame, '.curve'
       OPEN(UNIT=SPECTRA_DATA_HANDLE, FILE=FileName, STATUS='UNKNOWN')
    END IF

    Spectra=>FirstSpectra
    DO WHILE (ASSOCIATED(Spectra))        
       
       IF (Spectra%Level == MAXIMUMSPECTRALEVEL) THEN
          SpectraLevel = FinestLevel
          mB(:,1)=1
          mB(:,2)=levels(SpectraLevel)%mX
       ELSE
          SpectraLevel = Spectra%Level
          mB=Spectra%mB
       END IF

       IF (Spectra%type==SCALAR_SPECT) THEN
          CALL CreatePlan(plan, SpectraLevel, mB, 1)
       ELSE
          CALL CreatePlan(plan, SpectraLevel, mB, nDim)
       END IF

       
       CALL CreateLayout(leveldown(mB, SpectraLevel, 0), spectra_layout)
       spectra_layout%level=0
       mB=spectra_layout%mB(MPI_ID, :,:)
       ALLOCATE(data(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), size(spectra%fields)))
       CALL LoadFieldIntoLayout(spectra_layout, data, Spectra%Fields%iD,0)
       
       CALL WindowData(Spectra%WindowFunction, spectra_layout, data)
       DO i=0, SpectraLevel-1
          CALL ProlongateLayout(spectra_layout, data, Spectra%method, Spectra%Fields%iD)
          
          !The problem with loading and windowing is that coarse cells will be windowed multiple times.  To avoid this we need to store the unwindowed coarse cells so that after windowing we can restore their values.
          mB=spectra_layout%mB(MPI_ID, :,:)
          ALLOCATE(tempdata(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), size(spectra%fields)))
          tempdata=data !make a copy
          data=UNDEFINED !set the new data to be undefined
          CALL LoadFieldIntoLayout(spectra_layout, data, Spectra%Fields%iD, i+1)
          !Now only refined cells will be defined
          !So we will flag those as undefined in our backup array
          WHERE(data/=UNDEFINED) tempdata=UNDEFINED
          CALL WindowData(Spectra%WindowFunction, spectra_layout, data)
          !Then we will use defined cells in the backup array to update the new array
          WHERE(tempdata /= UNDEFINED) data=tempdata
          DEALLOCATE(tempdata)
       END DO
       
       CALL TransferLayouts(spectra_layout, plan%layouts(1)%p, data, plan%data)      

       IF (Spectra%method == SPECTRAL_PROLONGATION) THEN
          write(str,'(A4,I1)') 'plan', MPI_ID
          CALL WriteBOV3DScalar(str, REAL((/lbound(plan%data,1), lbound(plan%data,2), lbound(plan%data,3)/)-1d0, KIND=qPREC), REAL((/ubound(plan%data,1), ubound(plan%data,2), ubound(plan%data,3)/), KIND=qPREC), 0d0, REAL(plan%data(:,:,:,1)), 'dens')
       END IF

       CALL ExecutePlan(plan, BACKWARD)
          
       CALL BinSpectra(Spectra, Spectralevel, plan%mB, plan%lmB, plan%data)
       CALL DestroyPlan(plan)
       CALL CollectSpectra(Spectra)
       CALL OutputSpectra(Spectra)
       DEALLOCATE(Spectra%Data)
       Spectra=>Spectra%next            
    END DO
    IF (MPI_ID == 0 .AND. ASSOCIATED(FirstSpectra)) CLOSE(SPECTRA_DATA_HANDLE)

  END SUBROUTINE ProcessSpectras

  SUBROUTINE CollectSpectra(Spectra)
    TYPE(SpectraDef), POINTER :: Spectra
    INTEGER :: iErr
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: totals
    IF (MPI_NP == 1) RETURN
    CALL MPI_ALLREDUCE(MPI_IN_PLACE, Spectra%Data, size(Spectra%Data), MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
  END SUBROUTINE CollectSpectra


  SUBROUTINE OutputSpectra(Spectra)
    TYPE(SpectraDef), POINTER :: Spectra
    INTEGER :: i,j
    CHARACTER(LEN=9), PARAMETER :: DATAFORMAT = '(2E25.16)'
    IF (MPI_ID == 0) THEN
       SELECT CASE(Spectra%Type)
       CASE(SCALAR_SPECT)
          WRITE(SPECTRA_DATA_HANDLE, '(A2,3A)') '# ', TRIM(GetName(Spectra%Fields(1)))
          WRITE(SPECTRA_DATA_HANDLE, DATAFORMAT) 0d0, Spectra%data(1,1)
          DO j=2,size(Spectra%Data,1)
             WRITE(SPECTRA_DATA_HANDLE, DATAFORMAT) real(j-1), Spectra%Data(j,1)
          END DO
          WRITE(SPECTRA_DATA_HANDLE, *)
          WRITE(SPECTRA_DATA_HANDLE, *)

       CASE(VECTOR_SPECT)
          DO i=1,nDim+3
             IF (i <= nDim) THEN
                WRITE(SPECTRA_DATA_HANDLE, '(A2,A)') '# ', TRIM(GetName(Spectra%Fields(i)))
             ELSEIF (i == nDim+1) THEN
                WRITE(SPECTRA_DATA_HANDLE, '(A2,2A)') '# ', TRIM(GetName(Spectra%Fields(1))), '_total'
             ELSEIF (i == nDim+2) THEN
                WRITE(SPECTRA_DATA_HANDLE, '(A2,2A)') '# ', TRIM(GetName(Spectra%Fields(1))), '_div'
             ELSEIF (i == nDim+3) THEN
                WRITE(SPECTRA_DATA_HANDLE, '(A2,2A)') '# ', TRIM(GetName(Spectra%Fields(1))), '_sol'
             END IF
             WRITE(SPECTRA_DATA_HANDLE, DATAFORMAT) 0d0, Spectra%data(1,i)
             DO j=2,size(Spectra%Data,1)
                WRITE(SPECTRA_DATA_HANDLE, DATAFORMAT) real(j-1), Spectra%Data(j,i)
             END DO
             WRITE(SPECTRA_DATA_HANDLE, *)
             WRITE(SPECTRA_DATA_HANDLE, *)
          END DO
       END SELECT
       
    END IF
  END SUBROUTINE OutputSpectra


  SUBROUTINE BinSpectra(Spectra, level, GmB, mB, data)
    TYPE(SpectraDef) :: Spectra
    INTEGER, DIMENSION(3,2) :: mB, Gmb
    COMPLEX(8), DIMENSION(:,:,:,:), POINTER :: data
    REAL(KIND=qPREC) :: kmax, kvec(3), k2, KE, KEc
    INTEGER :: nfields, nbins,i,j,k,bin, level
    COMPLEX(8) :: v(3), v2(3)
    kmax=maxval(levels(Level)%mx)*sqrt(REAL(nDim))
    nbins=ceiling(kmax)+1
    SELECT CASE(Spectra%type)
    CASE(SCALAR_SPECT)
       nFields=1
    CASE(VECTOR_SPECT)
       nFields=nDim+3 ! 1 for each component + 2 decompositions + total
    END SELECT
    ALLOCATE(Spectra%Data(nBins, nFields))
    Spectra%data=0
    DO i=mB(1,1), mB(1,2)
       DO j=mB(2,1), mB(2,2)
          DO k=mB(3,1), mB(3,2)
             kvec=SpectraK((/i,j,k/),Gmb)
             k2=sum(kvec**2)
             bin=nint(sqrt(k2))+1
!             write(*,*) kvec, bin
             SELECT CASE(Spectra%type)
             CASE(SCALAR_SPECT)
                Spectra%data(bin,1)=Spectra%data(bin,1)+abs(data(i,j,k,1)**2)
             CASE(VECTOR_SPECT)
                v(1:nDim)=data(i,j,k,1:nDim)
                v2(1:nDim)=abs(v(1:nDim)**2)
                KE=sum(v2(1:nDim))
                Spectra%data(bin,1:nDim)=Spectra%data(bin,1:nDim)+v2
                Spectra%data(bin,nDim+1)=Spectra%data(bin,nDim+1)+KE
                IF (ANY(kvec /= 0)) THEN ! Helmholtz decomposition
                   v=DOT_PRODUCT(kvec,v)*kvec/k2
                   KEc=sum(abs(v**2))
                   Spectra%data(bin,nDim+2)=Spectra%data(bin,nDim+2)+KEc
                   Spectra%data(bin,nDim+3)=Spectra%data(bin,nDim+3)+KE-KEc
                END IF
             END SELECT
          END DO
       END DO
    END DO
    Spectra%data=Spectra%data/product(REAL(levels(level)%mX)**2)
  END SUBROUTINE BinSpectra


  SUBROUTINE WindowData(WindowFunction, layout, data)
     INTEGER :: WindowFunction
     TYPE(LayoutDef), POINTER :: layout
     COMPLEX(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: data
     INTEGER, DIMENSION(3,2) :: mB !global bounds of layout
     REAL(KIND=qPREC), DIMENSION(3) :: origin !Origin of layout
     REAL(KIND=qPREC), DIMENSION(3) :: radius !semi-principal axis
     INTEGER :: i,j,k
     REAL(KIND=qPREC) :: x,y,z,r,weight
     mB(:,1)=minval(layout%mB(:,:,1),1)
     mB(:,2)=maxval(layout%mB(:,:,2),1)
     origin=half*real(mB(:,1)+mB(:,2))
     radius=real((mB(:,2)-mB(:,1)+1))/2d0
     SELECT CASE (WindowFunction)
     CASE(COSINE_WINDOW)
        DO i=layout%mB(MPI_ID,1,1), layout%MB(MPI_ID,1,2)
           x=(REAL(i,KIND=qPREC)-origin(1))/radius(1)
           DO j=layout%MB(MPI_ID,2,1), layout%MB(MPI_ID,2,2)
              y=(REAL(j,KIND=qPREC)-origin(2))/radius(2)
              DO k=layout%MB(MPI_ID,3,1), layout%MB(MPI_ID,3,2)
                 z=(REAL(k,KIND=qPREC)-origin(3))/radius(3)
                 r=sqrt(x**2+y**2+z**2)
                 IF (r > 1d0) THEN
                    weight=0d0
                 ELSE
                    weight=cos(half*Pi*r)
                 END IF
                 data(i,j,k,:)=data(i,j,k,:)*weight
              END DO
           END DO
        END DO
     END SELECT
  END SUBROUTINE WindowData


END MODULE Spectras
