!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    splines.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Splines
   USE GlobalDeclarations
   USE PhysicsDeclarations

   IMPLICIT NONE

   TYPE SplineDef
      REAL(KIND=qPREC), POINTER, DIMENSION(:) :: x=>NULL()
      REAL(KIND=qPREC), POINTER, DIMENSION(:) :: y=>NULL()
      REAL(KIND=qPREC), POINTER, DIMENSION(:) :: b=>NULL()
      REAL(KIND=qPREC), POINTER, DIMENSION(:) :: c=>NULL()
      REAL(KIND=qPREC), POINTER, DIMENSION(:) :: d=>NULL()
   END type SplineDef

   TYPE pSplineDef
      TYPE(SplineDef), POINTER :: p
   END type pSplineDef

   ! Example of Usage:
   ! Call CreateSpline(Spline, 10)  !10 points
   ! Spline%x(1:10)=REAL((/(i,i=1,10)/))   !Initialize x values
   ! Spline%y=Spline%x**3+2d0*Spline%x**2   !Initialize y values
   ! CALL SolveSpline(Spline)
   ! write(*,*) 'value at x=1.5 is ' , GetSplineValue(Spline, 1.5d0)
   ! CALL DestroySpline(Spline)

CONTAINS


   SUBROUTINE CreateSpline(Spline, size)
      TYPE(SplineDef), POINTER :: Spline
      INTEGER :: size
      ALLOCATE(Spline)
      ALLOCATE(Spline%x(size))
      ALLOCATE(Spline%y(size))
   END SUBROUTINE CreateSpline

   SUBROUTINE DestroySpline(Spline)
      TYPE(SplineDef), POINTER :: Spline
      IF (ASSOCIATED(Spline%x)) DEALLOCATE(Spline%x)
      IF (ASSOCIATED(Spline%y)) DEALLOCATE(Spline%y)
      IF (ASSOCIATED(Spline%b)) DEALLOCATE(Spline%b)
      IF (ASSOCIATED(Spline%c)) DEALLOCATE(Spline%c)
      IF (ASSOCIATED(Spline%d)) DEALLOCATE(Spline%d)
      DEALLOCATE(Spline)
      NULLIFY(Spline)
   END SUBROUTINE DestroySpline


   !  Calculate the coefficients b(i), c(i), and d(i), i=1,2,...,n for cubic spline interpolation
   !  s(x) = y(i) + b(i)*(x-x(i)) + c(i)*(x-x(i))**2 + d(i)*(x-x(i))**3
   SUBROUTINE SolveSpline (Spline)
      TYPE(SplineDef) :: Spline
      INTEGER i, j, n
      REAL(KIND=qPREC) :: h
      REAL(KIND=qPREC), DIMENSION(:), POINTER :: b,c,d,x,y
      n=size(Spline%x)
      ALLOCATE(Spline%b(n), Spline%c(n), Spline%d(n))
      b=>Spline%b
      c=>Spline%c
      d=>Spline%d
      x=>Spline%x
      y=>Spline%y


      IF ( n < 2 ) RETURN !Need at least two points
      IF ( n < 3 ) THEN !Do linear interpolation
         b(1) = (y(2)-y(1))/(x(2)-x(1))   
         c(1) = 0d0
         d(1) = 0d0
         b(2) = b(1)
         c(2) = 0d0
         d(2) = 0d0
         RETURN
      END IF

      d(1) = x(2) - x(1)
      c(2) = (y(2) - y(1))/d(1)
      DO i = 2, n-1
         d(i) = x(i+1) - x(i)
         b(i) = 2d0*(d(i-1) + d(i))
         c(i+1) = (y(i+1) - y(i))/d(i)
         c(i) = c(i+1) - c(i)
      END DO

      b(1) = -d(1)
      b(n) = -d(n-1)
      c(1) = 0d0
      c(n) = 0d0
      IF(n /= 3) THEN
         c(1) = c(3)/(x(4)-x(2)) - c(2)/(x(3)-x(1))
         c(n) = c(n-1)/(x(n)-x(n-2)) - c(n-2)/(x(n-1)-x(n-3))
         c(1) = c(1)*d(1)**2/(x(4)-x(1))
         c(n) = -c(n)*d(n-1)**2/(x(n)-x(n-3))
      END IF

      DO i = 2, n
         h = d(i-1)/b(i-1)
         b(i) = b(i) - h*d(i-1)
         c(i) = c(i) - h*c(i-1)
      END DO

      c(n) = c(n)/b(n)
      DO j = 1, n-1
         i = n-j
         c(i) = (c(i) - d(i)*c(i+1))/b(i)
      END DO

      b(n) = (y(n) - y(n-1))/d(n-1) + d(n-1)*(c(n-1) + 2d0*c(n))
      DO i = 1, n-1
         b(i) = (y(i+1) - y(i))/d(i) - d(i)*(c(i+1) + 2d0*c(i))
         d(i) = (c(i+1) - c(i))/d(i)
         c(i) = 3d0*c(i)
      END DO
      c(n) = 3d0*c(n)
      d(n) = d(n-1)
   END SUBROUTINE SolveSpline


   !> Returns the interpolated value at position u
   FUNCTION GetSplineValue(Spline, u)
      TYPE(SplineDef) :: Spline
      REAL(KIND=qPREC) :: GetSplineValue
      INTEGER i, j,k, n
      REAL(KIND=qPREC) :: h, u
      REAL(KIND=qPREC), DIMENSION(:), POINTER :: b,c,d,x,y
      b=>Spline%b
      c=>Spline%c
      d=>Spline%d
      x=>Spline%x
      y=>Spline%y
      n=size(x)
      IF (n == 1) GetSplineValue=y(1)

      IF(u <= x(1)) THEN
         GetSplineValue = y(1)
         RETURN
      END IF
      IF(u >= x(n)) THEN
         GetSplineValue = y(n)
         RETURN
      END IF

      i = 1
      j = n+1
      DO WHILE (j > i+1)
         k = (i+j)/2
         IF(u < x(k)) THEN
            j=k
         ELSE
            i=k
         END IF
      END DO

      h = u - x(i)
      GetSplineValue = y(i) + h*(b(i) + h*(c(i) + h*d(i)))
   END FUNCTION GetSplineValue

END MODULE Splines
