!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    totals.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Totals
   USE TreeDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE ParticleDeclarations
   USE Fields
   USE Shapes
   IMPLICIT NONE
   TYPE TotalDef
      TYPE(FieldDef) :: Field
      REAL(KIND=qPREC) :: CurrentValue=0d0
      TYPE(ShapeDef), POINTER :: shape=>NULL()
      TYPE(TotalDef), POINTER :: next=>NULL()
   END TYPE TotalDef

   TYPE(TotalDef), POINTER :: FirstTotal, LastTotal
   LOGICAL :: lFirstTotalFrame
   SAVE
CONTAINS
   SUBROUTINE TotalInit()
      Nullify(FirstTotal, LastTotal)
      lFirstTotalFrame=.true.
   END SUBROUTINE TotalInit


   SUBROUTINE AddAllTotals(component, OptShape)
      INTEGER :: component, j
      TYPE(TotalDef), POINTER :: Total
      TYPE(ShapeDef), POINTER, OPTIONAL :: OptShape
      IF (Component /= GASCOMP .AND. Component /= PARTICLECOMP .AND. Component /= BOTHCOMP) THEN
         IF (MPI_ID == 0) PRINT*, 'Err - unrecognized component in AddAllTotals'
      ELSE
         DO j=1,NrFieldVars
            CALL CreateTotal(Total)
            Total%Field%Component=component
            Total%Field%id=j
            Total%Field%name=FieldName(j)
            IF (Present(OptShape)) Total%shape=>OptShape
         END DO
         DO j=1,NrTracerVars
            CALL CreateTotal(Total)
            Total%Field%Component=COMPONENT
            Total%Field%id=NrFieldVars+j
            Total%Field%name=TracerName(j)
            IF (Present(OptShape)) Total%shape=>OptShape
         END DO
         CALL CreateTotal(Total)
         Total%Field%Component=Component
         Total%Field%id=KE_Field
         Total%Field%name='Kinetic_Energy'
         IF (Present(OptShape)) Total%shape=>OptShape
         CALL CreateTotal(Total)
         Total%Field%Component=Component
         Total%Field%id=iE_Field
         Total%Field%name='Internal_Energy'
         IF (Present(OptShape)) Total%shape=>OptShape
         CALL CreateTotal(Total)
         Total%Field%Component=Component
         Total%Field%id=P_Field
         Total%Field%name='Pressure'
         IF (Present(OptShape)) Total%shape=>OptShape
         CALL CreateTotal(Total)
         Total%Field%Component=Component
         Total%Field%id=Temp_Field
         Total%Field%name='Temperature'
         IF (Present(OptShape)) Total%shape=>OptShape
         IF (lMHD) THEN
            CALL CreateTotal(Total)
            Total%Field%Component=Component
            Total%Field%id=BE_Field
            Total%Field%name='Magnetic_Energy'
            IF (Present(OptShape)) Total%shape=>OptShape
         END IF
         IF (lSelfGravity) THEN
            CALL CreateTotal(Total)
            Total%Field%Component=Component
            Total%Field%id=GravEnergy_Field
            Total%Field%name='Gravitational_Energy'
            IF (Present(OptShape)) Total%shape=>OptShape
         END IF
      END IF
   END SUBROUTINE AddAllTotals

   SUBROUTINE CreateTotal(Total)
      TYPE(TotalDef), POINTER :: Total
      ALLOCATE(Total)
      CALL AddTotalToList(Total)
   END SUBROUTINE CreateTotal

   SUBROUTINE DestroyTotal(Total)
      TYPE(TotalDef), POINTER :: Total
      DEALLOCATE(Total)
      NULLIFY(Total)
   END SUBROUTINE DestroyTotal

   SUBROUTINE AddTotalToList(Total)
      TYPE(TotalDef),POINTER :: Total
      IF(.NOT. ASSOCIATED(FirstTotal)) THEN ! First Total Object only
         FirstTotal=>Total
         LastTotal=>Total
      ELSE
         LastTotal%next=>Total
         LastTotal=>Total
      END IF
   END SUBROUTINE AddTotalToList

   SUBROUTINE ProcessTotals
      TYPE(TotalDef), POINTER :: Total
      Total=>FirstTotal
      DO WHILE (ASSOCIATED(Total))        
         CALL ProcessTotal(Total)
         Total=>Total%next            
      END DO
      CALL OutputTotals

   END SUBROUTINE ProcessTotals

   SUBROUTINE ProcessTotal(Total)
      TYPE(TotalDef), POINTER :: Total
      INTEGER :: n
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(ParticleListDef), POINTER :: ParticleList
      REAL(KIND=qPREC) :: rpos(3)
      Total%CurrentValue=0d0
      IF (Total%Field%Component==GASCOMP .OR. Total%Field%Component==BOTHCOMP) THEN
         DO n=0, MaxLevel
            Nodelist=>Nodes(n)%p
            DO WHILE (ASSOCIATED(NodeList))
               CALL ProcessTotalInfo(Nodelist%self%info, Total)
               Nodelist=>Nodelist%next
            END DO
         END DO
         CALL CollectTotal(Total)
      END IF
      IF (Total%Field%Component==PARTICLECOMP .OR. Total%Field%Component==BOTHCOMP) THEN
         ParticleList=>SinkParticles
         DO WHILE (ASSOCIATED(ParticleList))
            IF (ASSOCIATED(Total%shape)) THEN
               IF (IsInShape(Total%shape, ParticleList%self%xloc, rpos, levels(0)%tnow)) THEN
                  Total%CurrentValue=Total%CurrentValue+GetParticleField(ParticleList%self, Total%Field%id)
               END IF
            ELSE
               Total%CurrentValue=Total%CurrentValue+GetParticleField(ParticleList%self, Total%Field%id)
            END IF
            ParticleList=>ParticleList%next
         END DO
      END IF
   END SUBROUTINE ProcessTotal

   SUBROUTINE ProcessTotalInfo(Info, Total)
      TYPE(InfoDef) :: Info
      TYPE(TotalDef) :: Total
      INTEGER :: i,j,k
      REAL(KIND=qPREC) :: dv, pos(3), rpos(3)
      dv=levels(Info%level)%dx**nDim
      DO i=1, Info%mX(1)
         IF (iCylindrical /= NoCyl) dv=levels(Info%level)%dx**nDim*(Info%xBounds(1,1)+levels(Info%level)%dx*(REAL(i)-.5d0))
         DO j=1, Info%mX(2)
            DO k=1, Info%mX(3)
               IF (ASSOCIATED(Total%shape)) THEN
                  pos=CellPos(Info, i, j, k)
                  IF (.NOT. IsInShape(Total%shape, pos, rpos, levels(Info%level)%tnow)) CYCLE
               END IF
               IF (Info%level < MaxLevel) THEN
                  IF (Info%ChildMask(i,j,k) /= 0) THEN
                     !                     PRINT*, 'cycling'
                     CYCLE
                  END IF
               END IF
               Total%CurrentValue=Total%CurrentValue+GetField(Info,i,j,k,Total%Field%ID)*dv
            END DO
         END DO
      END DO
   END SUBROUTINE ProcessTotalInfo



   SUBROUTINE CollectTotal(Total)
      TYPE(TotalDef), POINTER :: Total
      INTEGER :: iErr
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, Total%CurrentValue, 1, MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
   END SUBROUTINE CollectTotal


   SUBROUTINE OutputTotals
      TYPE(TotalDef), POINTER :: Total
      INTEGER :: iErr
      IF (MPI_ID == 0) THEN
         OPEN(UNIT=TOTALS_DATA_HANDLE, FILE=TOTALS_DATA_FILE, STATUS='OLD', POSITION='APPEND', IOSTAT=iErr)
         IF (iErr /= 0) THEN !File does not already exist
            OPEN(UNIT=TOTALS_DATA_HANDLE, FILE=TOTALS_DATA_FILE, STATUS='NEW', IOSTAT=iErr)
            IF (iErr /= 0) THEN !File cannot be created new either
               PRINT*, 'Error - cannot open file as old or new'
               STOP
            END IF
         END IF
         IF (lFirstTotalFrame) THEN
            write(TOTALS_DATA_HANDLE,'(A20)', ADVANCE='NO') 'Time                '
            Total=>FirstTotal
            DO WHILE (ASSOCIATED(Total))
               IF (ASSOCIATED(total%next)) THEN
                  write(TOTALS_DATA_HANDLE,'(A20)', ADVANCE='NO') GetName(Total%field)
               ELSE
                  write(TOTALS_DATA_HANDLE,'(A20)') GetName(Total%field)
               END IF
               Total=>Total%next
            END DO
            lFirstTotalFrame=.false.
         END IF
         write(TOTALS_DATA_HANDLE,'(E20.11)', ADVANCE='NO') levels(0)%tnow
         Total=>FirstTotal
         DO WHILE (ASSOCIATED(Total))
            IF (ASSOCIATED(total%next)) THEN
               write(TOTALS_DATA_HANDLE,'(E20.11)', ADVANCE='NO') Total%CurrentValue
            ELSE
               write(TOTALS_DATA_HANDLE,'(E20.11)') Total%CurrentValue
            END IF
            Total=>Total%next
         END DO
         CLOSE(TOTALS_DATA_HANDLE)
      END IF
   END SUBROUTINE OutputTotals
END MODULE Totals
