!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    cooling.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir source
!! @brief directory containing modules for handling source terms

!> @file cooling.f90
!! @brief Cooling source terms file

!! @ingroup Source

!> Module for handling cooling source terms
MODULE CoolingSrc

  USE DataDeclarations
  USE PhysicsDeclarations
  USE EOS
  USE SourceDeclarations
  USE NEQCoolingSrc

  IMPLICIT NONE
  PRIVATE
  PUBLIC Cooling,CreateCoolingObject,DestroyCoolingObject,InitCoolingTracers, InitCoolingElliptics, CoolingCheck, CoolingIO, CoolingFinalizeInit, Cooling_CountObjects
  PUBLIC Cooling_ReadObjectFromChombo, Cooling_InitChomboDatasets, Cooling_WriteObjectToChombo, GetIICoolEqTemp, InitIICool, IICoolingRate, GetCoolingStrength, GetIICoolEqDensity, InitZCool, GetZvars, ZCoolingRate, Zweight
  TYPE,PUBLIC :: CoolingDef
     ! analytic cooling alpha, beta
     REAL(KIND=qPrec) :: alpha,beta
     INTEGER :: iCooling    ! see parameters for iCooling below
     LOGICAL :: lPrimitive  ! primitive or conservative form of source term
     REAL(KIND=qPrec) :: FloorTemp ! do not apply cooling below this temperature
     REAL(KIND=qPrec) :: MinTemp   ! do not allow things to cool below this temperature
     !
     ! parameters to specify region to apply source term, *** spatially driven ***
     REAL(KIND=xPrec) :: xlower(3)=0d0,xupper(3)=0d0  ! rectangular region
     REAL(KIND=xPrec) :: pos(3),radius=0d0            ! spherical region
     ! parameters to specify range to apply source term, *** variable driven ***
     INTEGER,DIMENSION(:),ALLOCATABLE :: var
     REAL(KIND=qPrec),DIMENSION(:,:),ALLOCATABLE :: varRange
     ! scaling to computational units
     REAL(KIND=qPrec) :: ScaleCool
     ! in locations where two source terms overlap,
     ! one should have priority (more than two will not work right)
     LOGICAL :: lHasPriority
     ! linked-list bookeeping variables
     INTEGER :: id
     TYPE(CoolingDef),POINTER :: previous,next
  END TYPE CoolingDef

  ! Parameters for iCooling
  INTEGER,PARAMETER,PUBLIC :: NoCool=0, AnalyticCool=1, DMCool=2, IICool=3, NEQCool=4, ZCool=5

  TYPE(CoolingDef),PUBLIC,POINTER :: firstcoolingobj,lastcoolingobj
  INTEGER :: iCoolingObjID=0

  ! Cooling tables
  REAL(KIND=qPrec),DIMENSION(:),ALLOCATABLE :: IICoolingTab,lognetab,temptab,logxtab
  REAL(KIND=qPrec),DIMENSION(:,:,:),ALLOCATABLE :: ZCoolingTab

  ! Parameters for ZCooling.tab to be read in and used properly
  INTEGER :: nDensities, nTemps, nXs
  REAL(KIND=qPrec) ::  lognemin, lognemax, tempmin, tempmax, logxmin, logxmax
  REAL(KIND=qPREC), PUBLIC :: IICoolPar(1:7)=(/2d-26,1d7,1.184d5,1d3,1.4d-2,9.2d1,0d0/)
CONTAINS


  ! ==================================================================
  ! =                       Cooling IO                               =
  ! ==================================================================

  !> Cooling IO frontend subroutine
  !! @params i_IO Integer, 0==output, 1==input
  SUBROUTINE CoolingIO(i_IO)
    ! Interface declarations
    INTEGER,INTENT(IN) :: i_IO
    ! Internal declarations
    TYPE(CoolingDef),POINTER :: coolingobj

    IF(i_IO==0) THEN
       ! output
       CALL OutputCooling
    ELSE
       ! input
       CALL InputCooling
    END IF
  END SUBROUTINE CoolingIO

  !> Cooling output subroutine
  SUBROUTINE OutputCooling
    ! Internal declarations
    TYPE(CoolingDef),POINTER :: coolingobj

    coolingobj=>firstcoolingobj
    DO WHILE(ASSOCIATED(coolingobj))
       ! Do the output ...

       coolingobj=>coolingobj%next
    END DO
  END SUBROUTINE OutputCooling

  !> Cooling input subroutine
  SUBROUTINE InputCooling
    ! Internal declarations
    TYPE(CoolingDef),POINTER :: coolingobj

    ! Allocate object, nullify pointers, add to list of cooling objects
    CALL CreateCoolingObject(coolingobj)
    
    ! Do the input ...
  END SUBROUTINE InputCooling

  !> Returns the number of cooling objects in the firstcoolingobj list.
  INTEGER FUNCTION Cooling_CountObjects()

      TYPE(CoolingDef), POINTER :: coolingobj
      INTEGER :: counter

      coolingobj => firstcoolingobj
      counter = 0

      ! Count the number of cooling objects in this list.
      DO WHILE (ASSOCIATED(coolingobj))
          counter = counter + 1
          coolingobj => coolingobj%next
      END DO

      Cooling_CountObjects = counter

  END FUNCTION Cooling_CountObjects

  ! ==================================================================
  ! =                Main Cooling Section                            =
  ! ==================================================================


  !> Main cooling subroutine, loops over linked list and calls specific cooling source(s)
  !! @params q variable vector q
  !! @params dqdt update to variable vector q
  !! @params x location of current cell center
  !! @params dx size of current cell
  SUBROUTINE Cooling(q,dqdt,x,dx,lform)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec) :: dqdt(:),x(3),dx
    ! Internal declarations
    TYPE(CoolingDef),POINTER :: coolingobj
    LOGICAL :: lCool
    LOGICAL :: lform
    coolingobj=>firstcoolingobj
    DO WHILE(ASSOCIATED(coolingobj))
       ! first determine if cell is within cooling region/regime
       !lcool=.FALSE.
       !lcool=TestRegion(coolingobj,x,dx)
       !lcool=TestRegime(coolingobj,q)

       !IF(lCool) THEN
          SELECT CASE(coolingobj%iCooling)
          CASE(NoCool)
             ! do nothing
          CASE(AnalyticCool)
             CALL AnalyticCooling(q,dqdt,coolingobj,lform)
          CASE(DMCool)
             CALL DMCooling(q,dqdt,coolingobj,lform)
          CASE(IICool)
             CALL IICooling(q,dqdt,coolingobj,lform)
          CASE(NEQCool)
             CALL NEQCooling(q,dqdt,coolingobj,lform)
          CASE(ZCool)
             CALL ZCooling(q,dqdt,coolingobj,lform)
          CASE DEFAULT
          END SELECT
       !END IF

       coolingobj=>coolingobj%next
    END DO
  END SUBROUTINE Cooling

  FUNCTION GetCoolingStrength(q,lform)
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec) :: x(3),dx,GetCoolingStrength
    ! Internal declarations
    TYPE(CoolingDef),POINTER :: coolingobj
    LOGICAL :: lCool
    LOGICAL :: lform
    coolingobj=>firstcoolingobj
    GetCoolingStrength=0d0
    DO WHILE(ASSOCIATED(coolingobj))
       SELECT CASE(coolingobj%iCooling)
       CASE(NoCool)
          ! do nothing
       CASE(AnalyticCool)
          GetCoolingStrength=GetCoolingStrength+AnalyticCoolingStrength(q,coolingobj,lform)
       CASE(DMCool)
          GetCoolingStrength=GetCoolingStrength+DMCoolingStrength(q,coolingobj,lform)
       CASE(IICool)
          GetCoolingStrength=GetCoolingStrength+IICoolingStrength(q,coolingobj,lform)
       CASE(ZCool)
          GetCoolingStrength=GetCoolingStrength+ZCoolingStrength(q,coolingobj,lform)
       CASE DEFAULT
       END SELECT
       coolingobj=>coolingobj%next
    END DO

 END FUNCTION GetCoolingStrength


  !> Determines if cooling source term is to be applied in this region
  !! @param coolingobj Cooling source object
  !! @param x location of center of cell
  !! @param dx size of cell
  LOGICAL FUNCTION TestRegion(coolingobj,x,dx)
    TYPE(CoolingDef),POINTER :: coolingobj
    REAL(KIND=xPrec) :: x(3),dx
    TestRegion=.true.
    IF(ALL(coolingobj%xLower==coolingobj%xUpper) .AND. coolingobj%radius==0d0) RETURN
  END FUNCTION TestRegion

  !> Determines if cooling source term is to be applied in this regime
  !! @param coolingobj Cooling source object
  !! @param q variable vector q
  LOGICAL FUNCTION TestRegime(coolingobj,q)
    TYPE(CoolingDef),POINTER :: coolingobj
    REAL(KIND=qPrec)  :: q(:)
    TestRegime=.TRUE.
    IF(.NOT.(ALLOCATED(coolingobj%var))) RETURN
  END FUNCTION TestRegime
  


  ! ==================================================================
  ! =               Actual Cooling Source Terms                      =
  ! ==================================================================

  !>  Analytic cooling source term
  !! @details Analytic cooling source term, where the cooling
  !!          function is a single power law, alpha*T^beta
  !! @param q Fluid variable vector q
  !! @param dqdt Update to be applied to q
  !! @param coolingobj Cooling object
  SUBROUTINE AnalyticCooling(q,dqdt,coolingobj,lform)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec),INTENT(INOUT) :: dqdt(:)
    TYPE(CoolingDef),POINTER :: coolingobj
    LOGICAL :: lform
    ! Local declarations

    IF (lform .eqv. PRIMITIVE) THEN      
       dqdt(iE) = dqdt(iE)-AnalyticCoolingStrength(q,coolingobj,lform) * gamma1
    ELSE
       dqdt(iE) = dqdt(iE)-AnalyticCoolingStrength(q,coolingobj,lform)
    END IF
  END SUBROUTINE AnalyticCooling

  FUNCTION AnalyticCoolingStrength(q,coolingobj,lform)
     REAL(KIND=qPREC) :: AnalyticCoolingStrength
     REAL(KIND=qPrec) :: q(:)
     TYPE(CoolingDef),POINTER :: coolingobj
     ! Local declarations
     REAL(KIND=qPrec) :: P,T0,dqlocal,alpha,beta,qnew(NrVars),Pnew,Tnew,coolfrac
     LOGICAL :: lform

     alpha=coolingobj%alpha
     beta=coolingobj%beta
     P=MERGE(PrimPress(q),Press(q),lform .eqv. PRIMITIVE)
     T0 = P / q(1)
     !    write(*,*) alpha, beta, T0, TempScale, coolingobj%floortemp
     IF(T0*TempScale<coolingobj%floortemp) THEN
        AnalyticCoolingStrength=0d0
     ELSE
        AnalyticCoolingStrength=q(1)**2 * alpha*T0**beta*coolingobj%ScaleCool
     END IF
!     write(*,*) T0, AnalyticCoolingStrength
  END FUNCTION AnalyticCoolingStrength



  !> Dalgarno-McCray cooling source term
  !! @details Implementation of DM Cooling curve using piecewise-linear
  !!               reconstruction of the tabulated curve in log space
  !! @param q Fluid variable vector q
  !! @param dqdt Update to be applied to q
  !! @param coolingobj Cooling object
  SUBROUTINE DMCooling(q,dqdt,coolingobj,lform)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:),dqdt(:)
    TYPE(CoolingDef),POINTER :: coolingobj
    ! Local declarations
    REAL(KIND=qPrec) :: P,T0,dqlocal
    LOGICAL :: lform

    IF (lform .eqv. PRIMITIVE) THEN
       dqdt(iE) = dqdt(iE) - DMCoolingStrength(q,coolingobj,lform)*gamma1
    ELSE
       dqdt(iE) = dqdt(iE) - DMCoolingStrength(q,coolingobj,lform)
    END IF

  END SUBROUTINE DMCooling

  FUNCTION DMCoolingStrength(q,coolingobj,lform)
     REAL(KIND=qPrec) :: q(:),DMCoolingStrength
     TYPE(CoolingDef),POINTER :: coolingobj
     ! Local declarations
     REAL(KIND=qPrec) :: P,T0,dqlocal
     LOGICAL :: lform
     P=MERGE(PrimPress(q),Press(q),lform .eqv. PRIMITIVE)

     T0 = P/q(1) * TempScale
     IF (T0 < max(100d0,coolingobj%floortemp)) THEN
        DMCoolingStrength=0d0
     ELSE
        DMCoolingStrength=q(1)**(2) * DMCoolingRate(T0) * coolingobj%ScaleCool
     END IF
  END FUNCTION DMCoolingStrength

  !> Inoue-Inutsuka 2008 (arxiv only? http://arxiv.org/abs/0801.0486 ) cooling source term
  !! @details Implementation of II Cooling curve using piecewise-linear
  !!               reconstruction of the curve, itself tabulated from
  !!               a functional form
  !! XMU should be 1.27
  !! @param q Fluid variable vector q
  !! @param dqdt Update to be applied to q
  !! @param coolingobj Cooling object
  SUBROUTINE IICooling(q,dqdt,coolingobj,lform)
     ! Interface declarations
     REAL(KIND=qPrec) :: q(:),dqdt(:)
     TYPE(CoolingDef),POINTER :: coolingobj
     LOGICAL :: lform
     IF (lform .eqv. PRIMITIVE) THEN
        dqdt(iE)=dqdt(iE) + (IIHeatingStrength(q,coolingobj,lform)-IICoolingStrength(q,coolingobj,lform))*gamma1
     ELSE
        dqdt(iE)=dqdt(iE) + (IIHeatingStrength(q,coolingobj,lform)-IICoolingStrength(q,coolingobj,lform))
     END IF
  END SUBROUTINE IICooling

  FUNCTION IIHeatingStrength(q,coolingobj, lform)
     TYPE(CoolingDef),POINTER :: coolingobj
     REAL(KIND=qPREC) :: IIHeatingStrength, q(:)
     LOGICAL :: lform
     IIHeatingStrength=q(1)*nScale*IICoolPar(1) * coolingobj%ScaleCool
  END FUNCTION IIHeatingStrength

  FUNCTION IICoolingStrength(q,coolingobj,lform)
     TYPE(CoolingDef),POINTER :: coolingobj
     REAL(KIND=qPrec) :: P,T,IICoolingStrength, q(:)
     LOGICAL :: lform
     P=MERGE(PrimPress(q),Press(q), lform .eqv. PRIMITIVE)
     T = P/q(1)*TempScale
     IF(T<coolingobj%floortemp) THEN
        IICoolingStrength=0d0
     ELSE
        IICoolingStrength=q(1)**2*nScale**2*IICoolingRate(T)*coolingobj%ScaleCool
     END IF
  END FUNCTION IICoolingStrength

  !> rho is in part/cc and returns a temp in K
  FUNCTION GetIICoolEqTemp(rho)

     REAL(KIND=qPrec) :: logtemp,rho, GetIICoolEqTemp
     INTEGER :: i
     logtemp=4 !(0-8)
     !using .9 allows for a range of +- 10 in log space
     !If we want 16 digits of accuracy we need -16/log10(.9)=718
     DO i=1, 350
        IF (rho*IIHeatingRate() - rho**2*IICoolingRate(10d0**logtemp) > 0d0) THEN
           logtemp=logtemp+.9d0**i
!           write(*,*) 'a'
        ELSE
!           write(*,*) 'b', 1.2/(1.2**i)
           logtemp=logtemp-.9d0**i
        END IF
!        IF (MPI_ID == 0) write(*,*) 'adjusted temp to', 10**logTemp, rho, IICoolingRate(10d0**logtemp), logtemp
     END DO
     GetIICoolEqTemp=10d0**logTemp
  END FUNCTION GetIICoolEqTemp


  !> p is in units of Kelvin/cc and returns a density in part/cc
  FUNCTION GetIICoolEqDensity(p)
     REAL(KIND=qPrec) :: logrho,p, GetIICoolEqDensity
     INTEGER :: i
     logrho=7d0
     !using .1 allows for a range of +- 10 in log space
     !If we want 16 digits of accuracy we need -16/log10(.9)=360
      DO i=1, 350 !raise lower bounds to avoid missing local minima
        IF (10d0**logrho*IIHeatingRate() - (10d0**logrho)**2*IICoolingRate(p/10d0**logrho)  > 0d0) THEN
           logrho=logrho-.9d0**i
!           write(*,*) 'a', 10d0**logrho,10d0**logrho*Y * ( 1d0 - 10d0**logrho*IICoolingRate(p/10d0**logrho))
        ELSE
           logrho=logrho+.9d0**i
!           write(*,*) 'b', 10d0**logrho,10d0**logrho*Y * ( 1d0 - 10d0**logrho*IICoolingRate(p/10d0**logrho))
        END IF
!        IF (MPI_ID == 0) write(*,*) 'adjusted temp to', 10**logTemp, rho, IICoolingRate(10d0**logtemp), logtemp
     END DO
     GetIICoolEqDensity=10d0**logrho
   END FUNCTION GetIICoolEqDensity

  FUNCTION IICoolingRate(T)
     REAL(KIND=qPrec) :: T,IICoolingRate
     IICoolingRate=IICoolPar(1)*(IICoolPar(2)*exp(-IICoolPar(3)/(T+IICoolPar(4)))+IICoolPar(5)*sqrt(T)*exp(-IICoolPar(6)/max(1d0,T+IICoolPar(7))))
  END FUNCTION IICoolingRate


  FUNCTION IIHeatingRate()
     REAL(KIND=qPrec) :: IIHeatingRate
     IIHeatingRate=IICoolPar(1)
   END FUNCTION IIHeatingRate

  !> Non-equilibrium microphysics cooling source term
  !! @details Calculates ionization and recombination effects on cooling
  !!          and tracks number densities of H,H2,HI,He,HeII,HeIII
  !! @param q Fluid variable vector q
  !! @param dqdt Update to be applied to q
  !! @param coolingobj Cooling object
  SUBROUTINE NEQCooling(q,dqdt,coolingobj,lform)
    REAL(KIND=qPREC) :: q(:), dqdt(:)
    TYPE(CoolingDef), POINTER :: coolingobj
    LOGICAL :: lform

    IF (lform .eqv. PRIMITIVE) THEN
       dqdt(iE) = dqdt(iE) - NEQCoolingStrength(q,coolingobj,lform) * gamma1
    ELSE
       dqdt(iE) = dqdt(iE) - NEQCoolingStrength(q,coolingobj,lform)
    END IF

    dqdt(nSpeciesLo:nSpeciesHi) = dqdt(nSpeciesLo:nSpeciesHi) + NEQSpeciesChange(q,lform) 

  END SUBROUTINE NEQCooling

  FUNCTION NEQCoolingStrength(q,coolingobj,lform)
     REAL(KIND=qPrec) :: q(:), NEQCoolingStrength
     TYPE(CoolingDef), POINTER :: coolingobj
     LOGICAL :: lform
     REAL(KIND=qPrec) :: P, T0, mu
     REAL(KIND=qPrec), DIMENSION(0:nSpeciesHi) :: nvec
     REAL(KIND=qPrec), DIMENSION(NrHydroVars) :: f
     f = 0d0

     P = MERGE(PrimPress(q),Press(q),lform .eqv. PRIMITIVE)
     CALL GetNEQvars(q, mu, nvec)               ! defined in neqcooling.f90
     T0 = P/q(1) * TempScale * mu
  
     CALL Cool_Derivatives(q,f,T0,nvec)         ! defined in i_evolve.f90

     IF (T0 < coolingobj%floortemp) THEN
        NEQCoolingStrength = 0d0
     ELSE
        NEQCoolingStrength = f(iE)              ! scaling is done in i_evolve.f90
     END IF

  END FUNCTION NEQCoolingStrength



  !> ZCooling source term
  !! @details Uses Pat's cooling tables (forbidden lines between 2000-16500 K)
  !!          and uses NEQCooling to get more accurate electron density and 
  !!          ionization fraction
  !! @param q Fluid variable vector q
  !! @param dqdt Update to be applied to q
  !! @param coolingobj Cooling object
  SUBROUTINE ZCooling(q,dqdt,coolingobj,lform)
    REAL(KIND=qPREC) :: q(:), dqdt(:)
    TYPE(CoolingDef), POINTER :: coolingobj
    LOGICAL :: lform

    IF (lform .eqv. PRIMITIVE) THEN
       dqdt(iE) = dqdt(iE) - ZCoolingStrength(q,coolingobj,lform) * gamma1
    ELSE
       dqdt(iE) = dqdt(iE) - ZCoolingStrength(q,coolingobj,lform)
    END IF

    dqdt(nSpeciesLo:nSpeciesHi) = dqdt(nSpeciesLo:nSpeciesHi) + NEQSpeciesChange(q,lform) 

  END SUBROUTINE ZCooling

  FUNCTION ZCoolingStrength(q,coolingobj,lform)
     REAL(KIND=qPrec) :: q(:), ZCoolingStrength
     TYPE(CoolingDef),POINTER :: coolingobj
     REAL(KIND=qPrec) :: P,T0,nH,ne,x,mu
     LOGICAL :: lform
     REAL(KIND=qPrec), DIMENSION(0:nSpeciesHi) :: nvec
     REAL(KIND=qPrec), DIMENSION(NrHydroVars) :: f
     f = 0d0

     P = MERGE(PrimPress(q),Press(q),lform .eqv. PRIMITIVE)
     CALL GetNEQvars(q, mu, nvec)               ! defined in neqcooling.f90
     T0 = P/q(1) * TempScale * mu

     CALL Cool_Derivatives(q,f,T0,nvec,ZCool)         ! defined in i_evolve.f90
 
     CALL GetZvars(nvec, ne, x, nH)

     IF (T0 < coolingobj%floortemp) THEN
        ZCoolingStrength = 0d0
     ELSE
        IF(ZCoolingRate(ne,T0,x) == 0d0) THEN
           ZCoolingStrength = f(iE)
        ELSE
           ZCoolingStrength = f(iE) + (1d0-Zweight(T0))*metal_loss + Zweight(T0) * nH**2d0 * ZCoolingRate(ne,T0,x) * coolingobj%ScaleCool
        END IF
     END IF

  END FUNCTION ZCoolingStrength

  ! weight function used to average metal_loss and ZCoolingRate so that ZCoolingRate is heavily weighted
  ! in the middle of the Z cooling table, and goes to zero at the table limits. metal_loss is from NEQ
  ! cooling and is weighted in the opposited direction of ZCoolingRate.
  FUNCTION Zweight(T)
     REAL(KIND=qPrec) :: Zweight, T, midT
           
     midT = tempmax - (tempmax+tempmin)/2d0
     Zweight = (1d0/midT)**2d0 * (tempmax-T)*(T-tempmin) 
!     Zweight = 1d0   
 
  END FUNCTION Zweight

  FUNCTION NEQSpeciesChange(q,lform)
     REAL(KIND=qPrec) :: q(:), NEQSpeciesChange(nSpeciesLo:nSpeciesHi)
     LOGICAL :: lform
     REAL(KIND=qPrec) :: P,T0,mu
     REAL(KIND=qPrec), DIMENSION(0:nSpeciesHi) :: nvec
     REAL(KIND=qPrec), DIMENSION(NrHydroVars) :: f
     f = 0d0

     P = MERGE(PrimPress(q),Press(q),lform .eqv. PRIMITIVE)
     CALL GetNEQvars(q, mu, nvec)               ! defined in neqcooling.f90
     T0 = P/q(1) * TempScale * mu

     CALL Cool_Derivatives(q,f,T0,nvec)         ! defined in i_evolve.f90

     NEQSpeciesChange = f(nSpeciesLo:nSpeciesHi)

  END FUNCTION NEQSpeciesChange

  ! Gets electron density, ionization fraction, and hydrogen number density
  SUBROUTINE GetZvars(nvecin,neout,xout,nHout)
     REAL(KIND=qPrec), DIMENSION(0:nSpeciesHi), INTENT(IN) :: nvecin
     REAL(KIND=qPrec) :: nneuc, npart
     REAL(KIND=qPrec), INTENT(OUT) :: neout, xout, nHout

     nneuc = nvecin(iH2)+nvecin(iH)+nvecin(iHII)+nvecin(iHe)+nvecin(iHeII)+nvecin(iHeIII)
     neout = nvecin(iHII) + nvecin(iHeII) + 2d0*nvecin(iHeIII) + nmine*nneuc
     npart = nneuc + neout
     nHout = nvecin(iH) + nvecin(iHII)
     xout = nvecin(iHII)/nHout

  END SUBROUTINE GetZvars

  ! trilinear interpolation algorithm based on formula on Paul Bourke's website
  FUNCTION ZCoolingRate(nein,tempin,xin)
     REAL(KIND=qPrec) :: lognein,logxin,ZCoolingRate,rate1,rate2,rate3,rate4,rate5,rate6,rate7,rate8,lintempin
     REAL(KIND=qPrec), INTENT(IN) :: nein, tempin, xin
     REAL(KIND=qPrec) :: lognestep,tempstep,logxstep
     INTEGER :: lognelow, lognehigh, templow, temphigh, logxlow, logxhigh 

     lognein = LOG10(nein)
     logxin = LOG10(xin)
     lintempin = tempin

     ! Compute step sizes of table Zcooling.tab
     lognestep = (lognemax-lognemin)/REAL(nDensities-1,qPrec)
     tempstep = (tempmax-tempmin)/REAL(nTemps-1,qPrec)
     logxstep = (logxmax-logxmin)/REAL(nXs-1,qPrec)

     ! Scale "in" values to fit within integer ranges of ZCoolingTab
     lognein = (lognein-lognemin)/lognestep + 1d0
     lintempin = (lintempin-tempmin)/tempstep + 1d0
     logxin = (logxin-logxmin)/logxstep + 1d0

     ! If the point is outside table range, return 0 cooling (essentially reverts to only being NEQ cooling)
     IF(lognein < 1 .OR. lognein > nDensities .OR. lintempin < 1 .OR. lintempin > nTemps .OR. logxin < 1 .OR. logxin > nXs) THEN
        ZCoolingRate = 0d0
     ELSE

        ! Find integers nearest to (lognein,tempin,logxin)
        lognelow = FLOOR(lognein); lognehigh = CEILING(lognein);
        templow = FLOOR(lintempin); temphigh = CEILING(lintempin);
        logxlow = FLOOR(logxin); logxhigh = CEILING(logxin);

        ! Match 8 closest points with their corresponding rates
        rate1 = ZCoolingTab(lognelow,templow,logxlow)
        rate2 = ZCoolingTab(lognehigh,templow,logxlow)
        rate3 = ZCoolingTab(lognelow,temphigh,logxlow)
        rate4 = ZCoolingTab(lognelow,templow,logxhigh)
        rate5 = ZCoolingTab(lognehigh,templow,logxhigh)
        rate6 = ZCoolingTab(lognelow,temphigh,logxhigh)
        rate7 = ZCoolingTab(lognehigh,temphigh,logxlow)
        rate8 = ZCoolingTab(lognehigh,temphigh,logxhigh)

        ! With low values --> 0 and high values --> 1, scale "in" values accordingly
        lognein = lognein-lognelow; lintempin = lintempin-templow; logxin = logxin-logxlow;

        ! With scaled values, the 8 closest points become the vertices of a unit cube
        ! aligned with the origin, and the interpolation is much simpler
        ZCoolingRate = rate1*(1d0-lognein)*(1d0-lintempin)*(1d0-logxin)
        ZCoolingRate = ZCoolingRate + rate2*lognein*(1d0-lintempin)*(1d0-logxin)
        ZCoolingRate = ZCoolingRate + rate3*(1d0-lognein)*lintempin*(1d0-logxin)
        ZCoolingRate = ZCoolingRate + rate4*(1d0-lognein)*(1d0-lintempin)*logxin
        ZCoolingRate = ZCoolingRate + rate5*lognein*(1d0-lintempin)*logxin
        ZCoolingRate = ZCoolingRate + rate6*(1d0-lognein)*lintempin*logxin
        ZCoolingRate = ZCoolingRate + rate7*lognein*lintempin*(1d0-logxin)
        ZCoolingRate = ZCoolingRate + rate8*lognein*lintempin*logxin

     END IF
  END FUNCTION ZCoolingRate

  ! ==========================================
  ! =      Cooling creation/destruction      =
  ! =      and list manipulation functions   =
  ! ==========================================

  SUBROUTINE CreateCoolingObject(coolingobj,userid)
     ! Interface declarations
     !INTEGER :: dummy
     TYPE(CoolingDef),POINTER :: coolingobj
     INTEGER,OPTIONAL :: userid

!     IF(ASSOCIATED(coolingobj)) THEN
!        PRINT*,'cooling_source.f90::CreateCoolingObject error -- object already associated. Halting.'
!        STOP
!     END IF

     ALLOCATE(coolingobj)
     NULLIFY(coolingobj%previous)
     NULLIFY(coolingobj%next)
     IF(PRESENT(userid)) THEN
        coolingobj%id=userid
     ELSE
        iCoolingObjID=iCoolingObjID+1
        coolingobj%id=iCoolingObjID
     END IF
     coolingobj%lhaspriority=.true.

     ! setup defaults: analytic cooling T^0.5, conservative w/o diagnostic field
     coolingobj%lprimitive=.false.
     coolingobj%icooling=AnalyticCool
     coolingobj%alpha=1d0
     coolingobj%beta=5d-1
     coolingobj%floortemp=1d-1
     coolingobj%mintemp=1d-2

     CALL AddCoolingObjToList(coolingobj)
  END SUBROUTINE CreateCoolingObject

  SUBROUTINE DestroyCoolingObject(coolingobj,id)
     TYPE(CoolingDef),POINTER :: coolingobj
     INTEGER,OPTIONAL :: id

     IF(PRESENT(id)) THEN
        coolingobj=>firstcoolingobj
        DO WHILE(ASSOCIATED(coolingobj))
           IF(coolingobj%id==id) THEN
              EXIT
           ELSE
              coolingobj=>coolingobj%next
           END IF
        END DO
     END IF
     CALL RemoveCoolingObjFromList(coolingobj)
     DEALLOCATE(coolingobj)
     NULLIFY(coolingobj)
  END SUBROUTINE DestroyCoolingObject

  SUBROUTINE AddCoolingObjToList(coolingobj)
     TYPE(CoolingDef),POINTER :: coolingobj

     IF(.NOT. ASSOCIATED(firstcoolingobj)) THEN ! first cooling object only
        firstcoolingobj=>coolingobj
        lastcoolingobj=>coolingobj
     ELSE
        coolingobj%previous=>lastcoolingobj
        lastcoolingobj%next=>coolingobj
        lastcoolingobj=>coolingobj
     END IF
  END SUBROUTINE AddCoolingObjToList

  SUBROUTINE RemoveCoolingObjFromList(coolingobj)
     TYPE(CoolingDef),POINTER :: coolingobj

     IF(ASSOCIATED(coolingobj%previous)) THEN
        coolingobj%previous%next=>coolingobj%next
     ELSE
        firstcoolingobj=>coolingobj%next
     END IF

     IF(ASSOCIATED(coolingobj%next)) THEN
        coolingobj%next%previous=>coolingobj%previous
     ELSE
        lastcoolingobj=>coolingobj%previous
        NULLIFY(lastcoolingobj%next)
     END IF
  END SUBROUTINE RemoveCoolingObjFromList

  LOGICAL FUNCTION CoolingCheck()
     TYPE(CoolingDef),POINTER :: coolingobj
     coolingcheck=.FALSE.
     coolingobj=>firstcoolingobj
     DO WHILE(ASSOCIATED(coolingobj))
        ! We need just one of the objects to exist 
        ! to keep lsrc true
        IF(coolingobj%iCooling/=0) THEN
           !          IF(testregion(coolingobj,x,dx) .AND. testregime(coolingobj)) THEN
           CoolingCheck=.TRUE.
           !          END IF
        END IF
        coolingobj=>coolingobj%next
     END DO
  END FUNCTION CoolingCheck


  ! ==========================================
  ! =       Initialization functions         =
  ! ==========================================

  SUBROUTINE InitCoolingTracers(sources)
     ! Interface declarations
      TYPE(SourcesDef),POINTER :: sources
     ! Internal declarations
     TYPE(CoolingDef),POINTER :: coolingObj
     INTEGER :: i
     coolingobj=>firstcoolingobj
     DO WHILE(ASSOCIATED(coolingObj))
        SELECT CASE(coolingObj%iCooling)
        CASE(NoCool)
        CASE(AnalyticCool)
        CASE(DMCool)
        CASE(NEQCool)
           CALL Addneqtracers
        CASE(ZCool)
           CALL Addneqtracers
        END SELECT
        coolingobj=>coolingobj%next
     END DO
  END SUBROUTINE InitCoolingTracers

  SUBROUTINE InitCoolingElliptics(sources)
     ! Interface declarations
     TYPE(SourcesDef),POINTER :: sources
     ! Internal declarations
     TYPE(CoolingDef),POINTER :: coolingObj
     INTEGER :: i
     coolingobj=>firstcoolingobj
     DO WHILE(ASSOCIATED(coolingObj))
        SELECT CASE(coolingObj%iCooling)
        CASE(NoCool)
        CASE(AnalyticCool)
        CASE(DMCool)
        CASE(NEQCool)
        CASE(ZCool)
        END SELECT
        coolingobj=>coolingobj%next
     END DO
  END SUBROUTINE InitCoolingElliptics

  !> Finalize initialization of cooling sources, including
  !!  allocating relevant tables
  SUBROUTINE CoolingFinalizeInit
     TYPE(CoolingDef),POINTER :: coolingobj

     coolingobj=>firstcoolingobj
     DO WHILE(ASSOCIATED(coolingobj))
        SELECT CASE(coolingobj%iCooling)
        CASE(NoCool)
        CASE(AnalyticCool)
           CALL InitAnalyticCool(coolingobj)
        CASE(DMCool)
           CALL InitDMCool
           coolingobj%scalecool=nScale**2*TimeScale/pScale
        CASE(IICool)
           CALL InitIICool(coolingobj)
        CASE(NEQCool)
           CALL InitNeqCool
        CASE(ZCool)
           coolingobj%scalecool=TimeScale/pScale
           CALL InitNeqCool
           CALL InitZCool
        END SELECT
        coolingobj=>coolingobj%next
     END DO
  END SUBROUTINE CoolingFinalizeInit

  !> Finalize initialization of analytic cooling source
  SUBROUTINE InitAnalyticCool(coolingobj)
     TYPE(CoolingDef),POINTER :: coolingobj
     !For power laws that have the same strength as brehm strahlung at a given shock speed (cs*Mach) 
     !use alpha=4.76e-20 (ergs*cm^3/s/K^.5) * (3d0/16d0*TempScale*(cs*Mach)**2)**(.5d0)  * (3d0/16d0 * (cs*Mach)**2) **(-coolingobj%beta)

     !Note if beta=.5 this is just pure Brehmstrahlung and
     ! alpha = 4.76e-20 (ergs*cm^3/s/K^.5) * (TempScale)**(.5d0)
     !See Imamura, Wolff, and Durisen (1983) although their alpha is our beta and their lambda is our alpha

     !alpha should have units of ergs*cm^3/s/K^beta
     coolingobj%scalecool=nScale**2/(pScale/Timescale)
     write(*,*) 'scalecool=', coolingobj%scalecool
  END SUBROUTINE InitAnalyticCool

  !> Allocate and fill II cooling table, finalize
  !!  initialization of II cooling source
  SUBROUTINE InitIICool(coolingobj)
     TYPE(CoolingDef),POINTER :: coolingobj
     !
     INTEGER :: i,nInput
     REAL :: dummy

     coolingobj%ScaleCool = 1d0/(pScale/TimeScale)

!     Switched to analytic expression instead of using a table
!     IF(ALLOCATED(IICoolingTab)) RETURN
!     OPEN(UNIT=85,FILE='TABLES/IIcooling.tab')
!     READ(85,*) nInput
!     ALLOCATE(IICoolingTab(nInput))
!     DO i=1,nInput
!        READ(85,*) dummy, IICoolingTab(i)
!     END DO
!     CLOSE(85)
  END SUBROUTINE InitIICool

  !> Allocate and fill Z cooling table, finalize
  !! initialization of Z cooling source
  !
  ! The first line of ZCooling.tab needs to be correct for Zcooling routines to work
  ! It should be: nDensities nTemps nXs...for example: 40 30 21
  !
  SUBROUTINE InitZCool

     IF(ALLOCATED(ZCoolingTab)) RETURN
     OPEN(UNIT=85,FILE='TABLES/Zcooling.tab')
     READ(85,*) nDensities,nTemps,nXs,lognemin,lognemax,tempmin,tempmax,logxmin,logxmax
     ALLOCATE(ZCoolingTab(nDensities,nTemps,nXs))
     READ(85,*) ZCoolingTab
     CLOSE(85)
  END SUBROUTINE InitZCool

  !> Read the data for a single cooling object in from a Chombo file.
  !! @param chandle An active Chombo file handle object.
  !! @param cooling_object An allocated CoolingDef object.
  SUBROUTINE Cooling_ReadObjectFromChombo(chandle, cooling_object)

     USE ChomboDeclarations, ONLY: ChomboHandle
     USE HDF5Declarations, ONLY: Read_Slab_From_Dataset_Int, Read_Slab_From_Dataset_Double

     TYPE(ChomboHandle), POINTER :: chandle
     TYPE(CoolingDef), POINTER :: cooling_object

     INTEGER, DIMENSION(1), TARGET :: int_buffer_array
     REAL(KIND=qPrec), DIMENSION(1), TARGET :: dbl_buffer_array
     INTEGER, DIMENSION(:), POINTER :: int_buffer
     REAL(KIND=qPrec), DIMENSION(:), POINTER :: dbl_buffer


     int_buffer => int_buffer_array
     dbl_buffer => dbl_buffer_array

     int_buffer = 0
     dbl_buffer = 0.d0

     CALL Read_Slab_From_Dataset_Int("iCooling", &
          chandle%source_group_id, &
          int_buffer, &
          chandle%source_offset)

     cooling_object%iCooling = int_buffer(1)

     CALL Read_Slab_From_Dataset_Double("alpha", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%alpha = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("beta", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%beta = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Int("lPrimitive", &
          chandle%source_group_id, &
          int_buffer, &
          chandle%source_offset)

     cooling_object%lPrimitive = (int_buffer(1) /= 0)

     ! Write position components to datasets.  Be sure to index the arrays using the i:i 
     ! notation, as this passes a 1-element array to the subroutine instead of a scalar.
     CALL Read_Slab_From_Dataset_Double("cartesian_xlower", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xlower(1) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("cartesian_ylower", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xlower(2) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("cartesian_zlower", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xlower(3) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("cartesian_xupper", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xupper(1) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("cartesian_yupper", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xupper(2) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("cartesian_zupper", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%xupper(3) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("spherical_pos_x", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%pos(1) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("spherical_pos_y", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%pos(3) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("spherical_pos_z", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%pos(3) = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("spherical_radius", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%radius = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Double("ScaleCool", &
          chandle%source_group_id, &
          dbl_buffer, &
          chandle%source_offset)

     cooling_object%ScaleCool = dbl_buffer(1)

     CALL Read_Slab_From_Dataset_Int("lHasPriority", &
          chandle%source_group_id, &
          int_buffer, &
          chandle%source_offset)

     cooling_object%lHasPriority = (int_buffer(1) /= 0)

     chandle%source_offset = chandle%source_offset + 1

  END SUBROUTINE Cooling_ReadObjectFromChombo


  !> Creates datasets for cooling objects within a Chombo file.
  !! @param chandle A chombo file handle.
  !! @param obj_count The number of cooling objects for which space should be allocated.
  SUBROUTINE Cooling_InitChomboDatasets(chandle, obj_count)

     USE ChomboDeclarations, ONLY: ChomboHandle
     USE HDF5Declarations, ONLY: Initialize_HDF5_Dataset_Int, Initialize_HDF5_Dataset_Double

     TYPE(ChomboHandle), POINTER :: chandle
     INTEGER :: obj_count

     TYPE(CoolingDef), POINTER :: cooling_object
     INTEGER :: i_err
     INTEGER :: iFixed


     IF (.NOT. ASSOCIATED(chandle)) THEN
        PRINT *, "Cooling_InitChomboDatasets error::invalid Chombo handle."
        STOP
     END IF

     IF (obj_count < 0) THEN
        PRINT *, "Cooling_InitChomboDatasets error::invalid object count ", obj_count, "."
        STOP
     END IF

     ! Position coordinates
     CALL Initialize_HDF5_Dataset_Int("iCooling", chandle%source_group_id, obj_count)

     CALL Initialize_HDF5_Dataset_Double("alpha", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("beta", chandle%source_group_id, obj_count)

     ! Logical variable.
     CALL Initialize_HDF5_Dataset_Int("lPrimitive", chandle%source_group_id, obj_count)

     CALL Initialize_HDF5_Dataset_Double("FloorTemp", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("MinTemp", chandle%source_group_id, obj_count)

     ! Cartesian region coordinates, stored in the same fashion as the sink particle coordinates.
     ! I don't know if this is helpful or not.
     CALL Initialize_HDF5_Dataset_Double("cartesian_xlower", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("cartesian_xupper", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("cartesian_ylower", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("cartesian_yupper", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("cartesian_zlower", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("cartesian_zupper", chandle%source_group_id, obj_count)

     ! Spherical region coordinates, stored in the same fashion as the sink particle coordinates.
     ! I don't know if this is helpful or not.
     CALL Initialize_HDF5_Dataset_Double("spherical_pos_x", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("spherical_pos_y", chandle%source_group_id, obj_count)
     CALL Initialize_HDF5_Dataset_Double("spherical_pos_z", chandle%source_group_id, obj_count)

     ! Radius of spherical region (scalar).
     CALL Initialize_HDF5_Dataset_Double("spherical_radius", chandle%source_group_id, obj_count)

     CALL Initialize_HDF5_Dataset_Double("ScaleCool", chandle%source_group_id, obj_count)

     ! Logical variables.
     CALL Initialize_HDF5_Dataset_Int("lHasPriority", chandle%source_group_id, obj_count)

  END SUBROUTINE Cooling_InitChomboDatasets

  !> Writes the data from a single cooling object to a Chombo file.
  !! @param chandle An active Chombo handle.
  SUBROUTINE Cooling_WriteObjectToChombo(chandle, cooling_object)

     USE ChomboDeclarations, ONLY: ChomboHandle
     USE HDF5Declarations, ONLY: Write_Slab_To_Dataset_Int, Write_Slab_To_Dataset_Double

     TYPE(ChomboHandle), POINTER :: chandle
     TYPE(CoolingDef), POINTER :: cooling_object


     CALL Write_Slab_To_Dataset_Int("iCooling", &
          chandle%source_group_id, &
          (/ cooling_object%iCooling /), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("alpha", &
          chandle%source_group_id, &
          (/ cooling_object%alpha /), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("beta", &
          chandle%source_group_id, &
          (/ cooling_object%beta /), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Int("lPrimitive", &
          chandle%source_group_id, &
          (/ BoolToInt(cooling_object%lPrimitive) /), &
          chandle%source_offset)

     ! Write position components to datasets.  Be sure to index the arrays using the i:i 
     ! notation, as this passes a 1-element array to the subroutine instead of a scalar.
     CALL Write_Slab_To_Dataset_Double("cartesian_xlower", &
          chandle%source_group_id, &
          cooling_object%xlower(1:1), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("cartesian_ylower", &
          chandle%source_group_id, &
          cooling_object%xlower(2:2), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("cartesian_zlower", &
          chandle%source_group_id, &
          cooling_object%xlower(3:3), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("cartesian_xupper", &
          chandle%source_group_id, &
          cooling_object%xupper(1:1), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("cartesian_yupper", &
          chandle%source_group_id, &
          cooling_object%xupper(2:2), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("cartesian_zupper", &
          chandle%source_group_id, &
          cooling_object%xupper(3:3), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("spherical_pos_x", &
          chandle%source_group_id, &
          cooling_object%pos(1:1), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("spherical_pos_y", &
          chandle%source_group_id, &
          cooling_object%pos(2:2), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("spherical_pos_z", &
          chandle%source_group_id, &
          cooling_object%pos(3:3), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("spherical_radius", &
          chandle%source_group_id, &
          (/ cooling_object%radius /), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Double("ScaleCool", &
          chandle%source_group_id, &
          (/ cooling_object%ScaleCool /), &
          chandle%source_offset)

     CALL Write_Slab_To_Dataset_Int("lHasPriority", &
          chandle%source_group_id, &
          (/ BoolToInt(cooling_object%lHasPriority) /), &
          chandle%source_offset)

     chandle%source_offset = chandle%source_offset + 1

  END SUBROUTINE Cooling_WriteObjectToChombo
END MODULE CoolingSrc
