!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    cylindrical.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE CylindricalSrc

  USE DataDeclarations
  USE PhysicsDeclarations
  USE EOS
  USE SourceDeclarations

  IMPLICIT NONE
  PRIVATE
  PUBLIC Cylindrical

CONTAINS

  SUBROUTINE Cylindrical(q,dqdt,pos,lform)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec),INTENT(INOUT) :: dqdt(:)
    REAL(KIND=xPrec) :: pos(:)
    LOGICAL :: lform
!    RETURN
       SELECT CASE(lform)
       CASE(PRIMITIVE)
          CALL PrimCyl(q,dqdt,pos)
       CASE(CONSERVATIVE)
          CALL ConsCyl(q,dqdt,pos)
       END SELECT

  END SUBROUTINE Cylindrical

  !> Cylindrical source term, primitive variables formulation
  SUBROUTINE PrimCyl(q,dqdt,pos)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec),INTENT(INOUT) :: dqdt(:)
    REAL(KIND=xPrec) :: pos(:)
    ! Local declarations
    REAL(KIND=xPrec) :: r
    REAL(KIND=qPrec) :: P, vr, vz, vphi, Br, Bz, Bphi, uB
    REAL(KIND=qPrec),DIMENSION(NrVars) :: dqlocal

    ! local vars for readability

    r=pos(irDir)
    P=PrimPress(q)
    vr=q(imom(irdir))

    !--- Hydro contributions ---!
	 dqlocal=0d0
    dqlocal(1) = q(1)*vr
    IF(iE .NE. 0) dqlocal(iE) = vr*(q(iE)+P)
    IF(NrTracerVars>0) dqlocal(nTracerLO:nTracerHI) = vr*q(nTracerLO:nTracerHI)
    IF (iCylindrical==WithAngMom) THEN
       dqlocal(imom(irDir))=-q(m_high)**2
       dqlocal(m_high) = q(imom(irdir))*q(m_high)
    END IF

    !--- MHD contributions ---!
    IF(lMHD) THEN
       vz=q(imom(izdir))
       Br=q(iB(irdir))
       Bz=q(iB(izdir))
       Bphi=q(iBz)
       vphi=0d0
       IF(iCylindrical==WithAngMom) vphi=q(m_high)
       uB = vr*Br + vz*Bz + vphi*Bphi

       dqlocal(imom(izDir)) = -Br*Bz/q(1)
       IF(iE .NE. 0) dqlocal(iE) = dqlocal(iE) + Br*uB
       dqlocal(iBy) = -Br*vz + Bz*vr
       IF(iCylindrical==WithAngMom) THEN
          dqlocal(imom(irDir)) = dqlocal(imom(irDir)) - (Br**2-Bphi**2)/q(1)
          dqlocal(m_high) = dqlocal(m_high) - 2d0*Br*Bphi/q(1)
       END IF       
    END IF

    dqdt = dqdt + (-dqlocal/r)
  END SUBROUTINE PrimCyl


  !> Cylindrical source term, conservative variables formulation
  SUBROUTINE ConsCyl(q,dqdt,pos)!,cylobj)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:)
    REAL(KIND=qPrec),INTENT(INOUT) :: dqdt(:)
    REAL(KIND=xPrec) :: pos(:)
    ! Local declarations
    REAL(KIND=xPrec) :: r
    REAL(KIND=qPrec) :: P, vr, vz, vphi, Br, Bz, Bphi, Bsq, uB
    REAL(KIND=qPrec),DIMENSION(NrVars) :: dqlocal

    ! local vars for readability
    r=pos(irDir)

    P=Press(q)
    vr = q(imom(irdir))/q(1)

    dqlocal=0d0
    !--- Hydro contributions ---!
    dqlocal(1)     = q(imom(irdir))
    dqlocal(imom(irDir)) = vr*q(imom(irDir))
    dqlocal(imom(izDir)) = vr*q(imom(izdir))
    IF(iE .NE. 0) dqlocal(iE) = vr*(q(iE)+P)
    IF(NrTracerVars>0) dqlocal(nTracerLO:nTracerHI) = vr*q(nTracerLO:nTracerHI)
    IF(iCylindrical==WithAngMom) THEN
       dqlocal(m_high)  = 2.d0*vr*q(m_high)
       dqlocal(imom(irDir))    = dqlocal(imom(irDir)) - q(m_high)**2/q(1)
    END IF

    !--- MHD  contributions ---!
    IF(lMHD) THEN
       vz=q(imom(izdir))/q(1)
       vphi=0d0
       IF(iCylindrical==WithAngMom) vphi=q(m_high)/q(1)
       Br=q(iB(irdir))
       Bz=q(iB(izdir))
       Bphi=q(iBz)
       Bsq = Br**2 + Bz**2 + Bphi**2
       uB = vr*Br + vz*Bz + vphi*Bphi

       dqlocal(imom(irDir)) = dqlocal(imom(irDir)) - Br**2 + Bphi**2
       dqlocal(imom(izDir)) = dqlocal(imom(izDir)) - Br*Bz
       dqlocal(iBy) = ( vr*Bz - vz*Br )
       IF(iE .NE. 0) dqlocal(iE) = dqlocal(iE) + vr*half*Bsq - Br*uB
        IF(iCylindrical==WithAngMom) THEN
           dqlocal(m_high) = dqlocal(m_high) - 2d0*Bphi*Br
        END IF
    END IF

    dqdt = dqdt + (-dqlocal/r)
  END SUBROUTINE ConsCyl
END MODULE CylindricalSrc
