!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    outflow.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE OutflowSrc

  USE DataDeclarations
  USE PhysicsDeclarations
  USE SourceDeclarations

  IMPLICIT NONE
  PRIVATE

  !> Contains data for outflows
  TYPE, PUBLIC :: OutflowDef
     REAL(KIND=xPrec)    :: pos(3),radius,v0(3),t0,mass,massloss
     REAL(KIND=xPrec)    :: outflow_vector(3)
     REAL(KIND=qPrec)    :: opening_angle
     REAL(KIND=qPrec)    :: massflux,momentumflux,energyflux
     REAL(KIND=xPrec)    :: precession_angle   ! used to update outflow_vector
     REAL(KIND=qPrec)    :: precession_rate    !   do precession
     LOGICAL             :: lBipolar
     REAL(KIND=qPrec)    :: tstart, tend, startinterval, endinterval
     INTEGER             :: starttype, endtype
     !
     INTEGER,DIMENSION(:),ALLOCATABLE :: iTracerFields,iEllipticFields,iDiagnosticFields     
     TYPE(OutflowDef),POINTER :: previous,next ! used to handle linked list of outflow objects
     INTEGER             :: ID
  END TYPE OutflowDef

  PUBLIC Outflows,CreateOutflowObject,DestroyOutflowObject,QueryOutflows,GetOutflowByID,OutflowsIO

  !> Generic interface for querying outflows
  INTERFACE QueryOutflows
     MODULE PROCEDURE   queryInt, queryFloat, queryDouble, queryLogical, queryCh
  END INTERFACE

  ! parameters for start and end types
  INTEGER,PARAMETER :: instant=1,linear=2,exponential=3

  TYPE(OutflowDef),PUBLIC,POINTER :: FirstOutflow, LastOutflow
  INTEGER :: iOutflowID=0

CONTAINS



  !> Outflow(s) source term
  !! @params q fluid variables for a given cell
  !! @params dqdt change in fluid variables
  SUBROUTINE Outflows(q,dqdt,pos)
    ! Interface declarations
    REAL(KIND=qPrec) :: q(:),dqdt(:)
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: outflow
    REAL(KIND=xPrec) :: outflowpos(3),radius
    REAL(KIND=xPrec) :: outflow_vector(3)
    REAL(KIND=qPrec) :: opening_angle
    REAL(KIND=qPrec) :: massflux,momentumflux,energyflux
    REAL(KIND=xPrec) :: precession_angle ! used to update outflow_vector
    REAL(KIND=qPrec) :: precession_rate  !   in order to do precession
    LOGICAL          :: lBipolar
    !
    INTEGER i

    outflow=>firstoutflow
    DO WHILE(ASSOCIATED(outflow))
       outflowpos       = outflow%pos
       radius           = outflow%radius
       outflow_vector   = outflow%outflow_vector
       opening_angle    = outflow%opening_angle
       massflux         = outflow%massflux
       momentumflux     = outflow%momentumflux
       energyflux       = outflow%energyflux
       precession_angle = outflow%precession_angle
       precession_rate  = outflow%precession_rate
       lBipolar         = outflow%lBipolar

       ! Update dqdt here
       ! ...
       ! ...

       outflow=>outflow%next
    END DO
          
  END SUBROUTINE Outflows

  ! ==========================================
  ! =      Outflow creation/destruction      =
  ! =      and list manipulation section     =
  ! ==========================================

  !> Create a new outflow object and add it to the list
  !! @params outflow pointer to the new outflow object
  !! @params userid Optional: user-specified ID for this object
  SUBROUTINE CreateOutflowObject(outflow,userid)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    INTEGER,OPTIONAL :: userid
    ! Internal declarations

    IF(ASSOCIATED(outflow)) THEN
       PRINT*,'outflow_source.f90::CreateOutflowObject error -- outflow object already associated. Halting.'
       STOP
    END IF

    ALLOCATE(outflow)
    NULLIFY(outflow%previous)
    NULLIFY(outflow%next)
    IF(PRESENT(userid)) THEN
       outflow%ID=userid
    ELSE
       iOutflowID=iOutflowID+1 
       outflow%ID=iOutflowID
    END IF
    CALL AddOutflowToList(outflow)
  END SUBROUTINE CreateOutflowObject

  !> Destroy outflow object
  !! @params outflow outflow object to be destroyed (dummy if ID is present)
  !! @params id Optional: outflow ID of outflow to be destroyed
  SUBROUTINE DestroyOutflowObject(outflow,id)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    INTEGER,OPTIONAL :: id
    ! Internal declarations

    IF(PRESENT(id)) THEN
       outflow=>firstoutflow
       DO WHILE(ASSOCIATED(outflow))
          IF(outflow%id==id) THEN
             EXIT
          ELSE
             outflow=>outflow%next
          END IF
       END DO
    END IF
    CALL RemoveOutflowFromList(outflow)
    DEALLOCATE(outflow)
    NULLIFY(outflow)
  END SUBROUTINE DestroyOutflowObject

  !> Adds outflow to outflow list
  !! @param outflow outflow object to be added to list
  SUBROUTINE AddOutflowToList(outflow)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    ! Internal declarations

    IF(.NOT. ASSOCIATED(firstoutflow)) THEN    ! first outflow only
       firstoutflow=>outflow
       lastoutflow=>outflow
    ELSE
       outflow%previous=>lastoutflow
       lastoutflow%next=>outflow
       lastoutflow=>outflow
    END IF
  END SUBROUTINE AddOutflowToList

  !> Removes outflow from outflow list
  !! @param outflow outflow object to be removed from list
  SUBROUTINE RemoveOutflowFromList(outflow)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    ! Internal declarations

    IF(ASSOCIATED(outflow%previous)) THEN
       outflow%previous%next=>outflow%next
    ELSE
       firstoutflow=>outflow%next
    END IF

    IF(ASSOCIATED(outflow%next)) THEN
       outflow%next%previous=>outflow%previous
    ELSE
       lastoutflow=>outflow%previous
       NULLIFY(lastoutflow%next)
    END IF

  END SUBROUTINE RemoveOutflowFromList

  SUBROUTINE GetOutflowByID(outflow,id)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    INTEGER :: id
    ! Internal declarations

    outflow=>firstoutflow
    DO WHILE(ASSOCIATED(outflow))
       IF(outflow%id==id) EXIT
       outflow=>outflow%next
    END DO
  END SUBROUTINE GetOutflowByID

  ! ====================================
  ! =           Query section          =
  ! ====================================

  !> Integer query 
  !! @params queryChar character representing query field to match
  !! @params IDlist the list of outflow IDs to be returned
  !! @params iquery integer query value
  SUBROUTINE queryInt(queryChar,IDlist,iquery)
    ! Interface declarations
    CHARACTER(LEN=*) :: queryChar
    INTEGER,DIMENSION(:) :: IDlist
    INTEGER,DIMENSION(:) :: iquery
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: currOutflow
    INTEGER :: i=0
    LOGICAL :: lMatch
    IDlist=0
    currOutflow=>firstoutflow
    DO WHILE(ASSOCIATED(currOutflow))
       CALL queryOutflow(currOutflow,lmatch,queryChar,iquery=iquery)
       IF(lmatch) THEN
          i=i+1
          IDlist(i)=curroutflow%id
       END IF
       currOutflow=>currOutflow%next
    END DO
  END SUBROUTINE queryInt

  !> Float query
  !! @params queryChar character representing query field to match
  !! @params IDlist the list of outflow IDs to be returned
  !! @params fquery float query value
  SUBROUTINE queryFloat(queryChar,IDlist,fquery)
    ! Interface declarations
    CHARACTER(LEN=*) :: queryChar
    INTEGER,DIMENSION(:) :: IDlist
    REAL,DIMENSION(:) :: fquery
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: currOutflow
    INTEGER :: i=0
    LOGICAL :: lMatch
    IDlist=0
    currOutflow=>firstoutflow
    DO WHILE(ASSOCIATED(currOutflow))
       CALL queryOutflow(currOutflow,lmatch,queryChar,fquery=fquery)
       IF(lmatch) THEN
          i=i+1
          IDlist(i)=curroutflow%id
       END IF
       currOutflow=>currOutflow%next
    END DO
  END SUBROUTINE queryFloat

  !> Double query
  !! @params queryChar character representing query field to match
  !! @params IDlist the list of outflow IDs to be returned
  !! @params dquery double query value
  SUBROUTINE queryDouble(queryChar,IDlist,dquery)
    ! Interface declarations
    CHARACTER(LEN=*) :: queryChar
    INTEGER,DIMENSION(:) :: IDlist
    REAL(KIND=qPrec),DIMENSION(:) :: dquery
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: currOutflow
    INTEGER :: i=0
    LOGICAL :: lMatch
    IDlist=0
    currOutflow=>firstoutflow
    DO WHILE(ASSOCIATED(currOutflow))
       CALL queryOutflow(currOutflow,lmatch,queryChar,dquery=dquery)
       IF(lmatch) THEN
          i=i+1
          IDlist(i)=curroutflow%id
       END IF
       currOutflow=>currOutflow%next
    END DO
  END SUBROUTINE queryDouble

  !> Logical query
  !! @params queryChar character representing query field to match
  !! @params IDlist the list of outflow IDs to be returned
  !! @params lquery logical query value
  SUBROUTINE queryLogical(queryChar,IDlist,lquery)
    ! Interface declarations
    CHARACTER(LEN=*) :: queryChar
    INTEGER,DIMENSION(:) :: IDlist
    LOGICAL,DIMENSION(:) :: lquery
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: currOutflow
    INTEGER :: i=0
    LOGICAL :: lMatch
    IDlist=0
    currOutflow=>firstoutflow
    DO WHILE(ASSOCIATED(currOutflow))
       CALL queryOutflow(currOutflow,lmatch,queryChar,lquery=lquery)
       IF(lmatch) THEN
          i=i+1
          IDlist(i)=curroutflow%id
       END IF
       currOutflow=>currOutflow%next
    END DO
  END SUBROUTINE queryLogical

  !> Character query
  !! @params queryChar character representing query field to match
  !! @params IDlist the list of outflow IDs to be returned
  !! @params cquery character query value
  SUBROUTINE queryCh(queryChar,IDlist,cquery)
    ! Interface declarations
    CHARACTER(LEN=*) :: queryChar
    INTEGER,DIMENSION(:) :: IDlist
    CHARACTER(LEN=*),DIMENSION(:) :: cquery
    ! Internal declarations
    TYPE(OutflowDef),POINTER :: currOutflow
    INTEGER :: i=0
    LOGICAL :: lMatch
    IDlist=0
    currOutflow=>firstoutflow
    DO WHILE(ASSOCIATED(currOutflow))
       CALL queryOutflow(currOutflow,lmatch,queryChar,cquery=cquery)
       IF(lmatch) THEN
          i=i+1
          IDlist(i)=curroutflow%id
       END IF
       currOutflow=>currOutflow%next
    END DO
  END SUBROUTINE queryCh
    
  !> Returns lmatch=T if this outflow matches the query criterion
  !! @details This subroutine is the main subroutine that all the queryXXXX
  !!               funnel into. The queryOutflows interface exists to 
  !!               make things simpler for users. This main subroutine
  !!               exists to make things simpler for coders, keeping
  !!               the queries themselves all in one location.
  !! @params outflow outflow to be queried
  !! @params lmatch logical, true if the outflow matches the query
  !! @params dquery Optional: double query
  !! @params fquery Optional: float query
  !! @params iquery Optional: integer query
  !! @params lquery Optional: logical query
  !! @params cquery Optional: character query
  SUBROUTINE queryOutflow(outflow,lmatch,queryChar,dquery,fquery,iquery,lquery,cquery)
    ! Interface declarations
    TYPE(OutflowDef),POINTER :: outflow
    LOGICAL :: lmatch
    CHARACTER(LEN=*) :: queryChar
    REAL(KIND=qPrec),DIMENSION(:),OPTIONAL :: dquery
    REAL,DIMENSION(:),OPTIONAL :: fquery
    INTEGER,DIMENSION(:),OPTIONAL :: iquery
    LOGICAL,DIMENSION(:),OPTIONAL :: lquery
    CHARACTER(LEN=*),DIMENSION(:),OPTIONAL :: cquery
    ! Internal declarations

    queryChar=lowercase(queryChar)

    ! Doubles
    IF(PRESENT(dquery)) THEN
       SELECT CASE(queryChar)
       CASE('pos')
          IF(ALL(outflow%pos==dquery)) lmatch=.TRUE.
       CASE('radius')
          IF(ALL(outflow%radius==dquery)) lmatch=.TRUE.
       CASE('outflow_vector')
          IF(ALL(outflow%outflow_vector==dquery)) lmatch=.TRUE.
       CASE('opening_angle')
          IF(ALL(outflow%opening_angle==dquery)) lmatch=.TRUE.
       CASE('massflux')
          IF(ALL(outflow%massflux==dquery)) lmatch=.TRUE.
       CASE('momentumflux')
          IF(ALL(outflow%momentumflux==dquery)) lmatch=.TRUE.
       CASE('energyflux')
          IF(ALL(outflow%energyflux==dquery)) lmatch=.TRUE.
       CASE('precession_angle')
          IF(ALL(outflow%precession_angle==dquery)) lmatch=.TRUE.
       CASE('precession_rate')
          IF(ALL(outflow%precession_rate==dquery)) lmatch=.TRUE.
       CASE('tstart')
          IF(ALL(outflow%tstart==dquery)) lmatch=.TRUE.
       CASE('tend')
          IF(ALL(outflow%tend==dquery)) lmatch=.TRUE.
       CASE('startinterval')
          IF(ALL(outflow%startinterval==dquery)) lmatch=.TRUE.
       CASE('endinterval')
          IF(ALL(outflow%endinterval==dquery)) lmatch=.TRUE.
       CASE DEFAULT
          PRINT*,'outflow_source.f90::QueryOutflow error -- unknown dquery specified. Halting.'
          STOP
       END SELECT
    ! Floats
    ELSE IF(PRESENT(fquery)) THEN
       SELECT CASE(queryChar)
       CASE DEFAULT
          PRINT*,'outflow_source.f90::QueryOutflow error -- unknown fquery specified. Halting.'
          STOP 
      END SELECT
    ! Integers
    ELSE IF(PRESENT(iquery)) THEN
       SELECT CASE(queryChar)
       CASE('starttype')
          IF(ALL(outflow%starttype==iquery)) lmatch=.TRUE.
       CASE('endtype')
          IF(ALL(outflow%endtype==iquery)) lmatch=.TRUE.
       CASE('id')
          IF(ALL(outflow%id==iquery)) lmatch=.TRUE.
       CASE DEFAULT
          PRINT*,'outflow_source.f90::QueryOutflow error -- unknown iquery specified. Halting.'
          STOP
       END SELECT
    ! Logicals
    ELSE IF(PRESENT(lquery)) THEN
       SELECT CASE(queryChar)
       CASE('lbipolar')
          IF(ALL(outflow%lbipolar==lquery)) lmatch=.TRUE.
       CASE DEFAULT
          PRINT*,'outflow_source.f90::QueryOutflow error -- unknown lquery specified. Halting.'
          STOP
       END SELECT
    ! Characters
    ELSE IF(PRESENT(cquery)) THEN
       SELECT CASE(queryChar)
       CASE DEFAULT
          PRINT*,'outflow_source.f90::QueryOutflow error -- unknown cquery specified. Halting.'
          STOP
       END SELECT
    ELSE
       PRINT*,'outflow_source.f90::QueryOutflow error -- subroutine called missing argument. Halting.'
       STOP
    END IF

  CONTAINS
    FUNCTION lowercase(string)
      CHARACTER(LEN=*) :: string
      CHARACTER(LEN=26) :: UPPER='ABCDEFGHIJKLMNOPQRSTUVWXYZ' &
                         , lower='abcdefghijklmnopqrstuvwxyz'
      CHARACTER(LEN=LEN(string)) :: lowercase
      INTEGER :: i,ind,ind2

      ! convert to lowercase
      ind=0;ind2=0
      DO i=1,LEN(string)
         ind=SCAN(string,UPPER)
         IF(ind>0) THEN 
            ind2=INDEX(UPPER,string(ind:ind))
            string(ind:ind)=lower(ind2:ind2)
            ind=0;ind2=0
         END IF
      END DO
      lowercase=string
    END FUNCTION lowercase
  END SUBROUTINE queryOutflow
END MODULE OutflowSrc
