!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    pointgravity.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE PointGravitySrc
   USE HyperbolicDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE SourceDeclarations
   USE CommonFunctions
   IMPLICIT NONE
   PRIVATE
   PUBLIC :: PointGravity, PointGravity_inst,  CreatePointGravityObject, DestroyPointGravityObject, CheckParticlePositions, PointGravityInit, PointGravityPotential
   PUBLIC :: PointGravity_CountObjects, FindPointGravityObject
   PUBLIC :: PointGravity_InitChomboDatasets, PointGravity_WriteObjectToChombo, PointGravity_ReadObjectFromChombo

   TYPE, PUBLIC :: PointGravityDef
      REAL(KIND=qPREC), DIMENSION(3) :: x0 = 0
      REAL(KIND=qPREC) :: t0 = 0
      REAL(KIND=qPREC), DIMENSION(3) :: v0 = 0
		REAL(KIND=qPREC) :: alpha = 1d0
      REAL(KIND=qPREC) :: mass = 0
      REAL(KIND=qPrec) :: dmom(0:MaxDepth,3) = 0
      REAL(KIND=qPREC) :: soft_length = 0
      INTEGER :: soft_function = NOSOFT
      INTEGER :: id

      TYPE(PointGravityDef), POINTER :: previous, next
   END TYPE PointGravityDef

   TYPE(PointGravityDef),PUBLIC,POINTER :: FirstPointGravityObj, LastPointGravityObj
   INTEGER :: nPointGravityObjects
   
   !> Pointer Nesting Type
   TYPE, PUBLIC :: pPointGravityDef
      TYPE(PointGravityDef), POINTER :: p
   END TYPE pPointGravityDef

CONTAINS

   SUBROUTINE PointGravityInit()
      NULLIFY(FirstPointGravityObj, LastPointGravityObj)
      nPointGravityObjects=0
   END SUBROUTINE PointGravityInit

   SUBROUTINE CheckParticlePositions()
      TYPE(PointGravityDef), POINTER :: PointGravityObj
      IF (ANY(lHydroPeriodic)) THEN
         PointGravityObj=>FirstPointGravityObj
         DO WHILE (ASSOCIATED(PointGravityObj))
            WHERE (lHydroPeriodic) PointGravityObj%x0=PointGravityObj%x0-floor((PointGravityObj%x0+PointGravityObj%v0*(levels(MaxLevel)%tnow-PointGravityObj%t0) - GxBounds(:,1))/(GxBounds(:,2)-GxBounds(:,1)))*(GxBounds(:,2)-GxBounds(:,1))
            PointGravityObj=>PointGravityObj%next
         END DO
      END IF
   END SUBROUTINE CheckParticlePositions


   SUBROUTINE PointGravity(q,dt,x,t,dv, level,lform)
      USE CommonFunctions
      ! Interface declarations

      REAL(KIND=qPrec) :: q(:)
      REAL(KIND=qPrec) :: dt,x(3),dx,pos(3),t, pOffset(3),r2,r,f_grav(3), dv

      ! Internal declarations
      TYPE(PointGravityDef),POINTER :: PointGravityObj
      LOGICAL :: lCool,lform
      INTEGER :: ioffset(3,2),i,j,k, level

      ioffset=0      
      WHERE(lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=2 !Checks the next periodic versions to the left and right
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)

      PointGravityObj=>FirstPointGravityObj
      DO WHILE(ASSOCIATED(PointGravityObj))
         DO i=ioffset(1,1),ioffset(1,2)
            DO j=ioffset(2,1),ioffset(2,2)
               DO k=ioffset(3,1),ioffset(3,2)
                  pOffSet=(/i,j,k/)*(GxBounds(:,2)-GxBounds(:,1))
                  pos=x - (PointGravityObj%x0+pOffset+(t+.5d0*dt-PointGravityObj%t0)*PointGravityObj%v0)
                  f_grav=GravityForce(PointGravityObj%Mass, pos, PointGravityObj%soft_length, PointGravityObj%soft_function)
                  IF (lform .eqv. PRIMITIVE) THEN
                     q(imom(1:nDim))=q(imom(1:nDim))+f_grav(1:nDim)*dt
                  ELSE
                     q(imom(1:nDim))=q(imom(1:nDim))+f_grav(1:nDim)*q(1)*dt
                     IF (iE .ne. 0) q(iE)=q(iE)+SUM(f_grav(1:nDim)*q(imom(1:nDim)))*dt
                  END IF
                  PointGravityObj%dMom(Level,1:nDim)=PointGravityObj%dMom(Level,1:nDim)-f_grav(1:nDim)*q(1)*dt*dv
               END DO
            END DO
         END DO
         PointGravityObj=>PointGravityObj%next         

      END DO
   END SUBROUTINE PointGravity


   SUBROUTINE PointGravity_inst(q,dqdt,x,t,lform)
      USE CommonFunctions
      ! Interface declarations
      REAL(KIND=qPrec) :: q(:)
      REAL(KIND=qPrec) :: dqdt(:),x(3),dx,pos(3),t, pOffset(3),r2,r,f_grav(3)
      ! Internal declarations
      TYPE(PointGravityDef),POINTER :: PointGravityObj
      LOGICAL :: lCool,lform
      INTEGER :: ioffset(3,2),i,j,k

      ioffset=0      
      WHERE(lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=2 !Checks the next periodic versions to the left and right
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)

      PointGravityObj=>FirstPointGravityObj
      DO WHILE(ASSOCIATED(PointGravityObj))
         DO i=ioffset(1,1),ioffset(1,2)
            DO j=ioffset(2,1),ioffset(2,2)
               DO k=ioffset(3,1),ioffset(3,2)
                  pOffSet=(/i,j,k/)*(GxBounds(:,2)-GxBounds(:,1))
                  pos=x - (PointGravityObj%x0+pOffset+(t-PointGravityObj%t0)*PointGravityObj%v0)
                  f_grav=GravityForce(PointGravityObj%Mass, pos, PointGravityObj%soft_length, PointGravityObj%soft_function)
                  IF (lform .eqv. PRIMITIVE) THEN
                     dqdt(imom(1:nDim))=dqdt(imom(1:nDim))+f_grav(1:nDim)
                  ELSE
                     dqdt(imom(1:nDim))=dqdt(imom(1:nDim))+f_grav(1:nDim)*q(1)
                     IF (iE .ne. 0) dqdt(iE)=dqdt(iE)+SUM(f_grav(1:nDim)*q(imom(1:nDim)))
                  END IF
               END DO
            END DO
         END DO
!         write(*,*) PointGravityObj%Mass, PointGravityObj%x0, x-pos
         PointGravityObj=>PointGravityObj%next         

      END DO
!      STOP
   END SUBROUTINE PointGravity_inst



   FUNCTION PointGravityPotential(PointGravityObj, pos, t)
     TYPE(PointGravityDef) :: PointGravityObj
     REAL(KIND=qPREC) :: PointGravityPotential
     REAL(KIND=qPREC) :: t, rpos(3),pos(3)
     rpos=pos-(PointGravityObj%x0+PointGravityObj%v0*(t-PointGravityObj%t0))
     PointGravityPotential = GravityPotential(PointGravityObj%Mass, rpos, PointGravityObj%soft_length, PointGravityObj%soft_function)
     
   END FUNCTION PointGravityPotential

  ! ==========================================
  ! =      PointGravity creation/destruction      =
  ! =      and list manipulation functions   =
  ! ==========================================

  SUBROUTINE CreatePointGravityObject(PointGravityObj)
    ! Interface declarations
    !INTEGER :: dummy
    TYPE(PointGravityDef),POINTER :: PointGravityObj

    IF(ASSOCIATED(PointGravityObj)) THEN
       PRINT*,'PointGravity_source.f90::CreatePointGravityObject error -- Object already associated. Halting.'
       STOP
    END IF
    ALLOCATE(PointGravityObj)
    NULLIFY(PointGravityObj%previous)
    NULLIFY(PointGravityObj%next)
    nPointGravityObjects=nPointGravityObjects+1
    PointGravityObj%id=nPointGravityObjects
    CALL AddPointGravityObjToList(PointGravityObj)
  END SUBROUTINE CreatePointGravityObject

  SUBROUTINE DestroyPointGravityObject(PointGravityObj,id)
    TYPE(PointGravityDef),POINTER :: PointGravityObj
    INTEGER,OPTIONAL :: id

    IF(PRESENT(id)) THEN
       PointGravityObj=>FirstPointGravityObj
       DO WHILE(ASSOCIATED(PointGravityObj))
          IF(PointGravityObj%id==id) THEN
             EXIT
          ELSE
             PointGravityObj=>PointGravityObj%next
          END IF
       END DO
    END IF
    CALL RemovePointGravityObjFromList(PointGravityObj)
    DEALLOCATE(PointGravityObj)
    NULLIFY(PointGravityObj)
  END SUBROUTINE DestroyPointGravityObject

  SUBROUTINE AddPointGravityObjToList(PointGravityObj)
    TYPE(PointGravityDef),POINTER :: PointGravityObj

    IF(.NOT. ASSOCIATED(FirstPointGravityObj)) THEN ! First PointGravity Object only
       FirstPointGravityObj=>PointGravityObj
       LastPointGravityObj=>PointGravityObj
    ELSE
       PointGravityObj%previous=>LastPointGravityObj
       LastPointGravityObj%next=>PointGravityObj
       LastPointGravityObj=>PointGravityObj
    END IF
  END SUBROUTINE AddPointGravityObjToList

  SUBROUTINE RemovePointGravityObjFromList(PointGravityObj)
    TYPE(PointGravityDef),POINTER :: PointGravityObj

    IF(ASSOCIATED(PointGravityObj%previous)) THEN
       PointGravityObj%previous%next=>PointGravityObj%next
    ELSE
       FirstPointGravityObj=>PointGravityObj%next
    END IF

    IF(ASSOCIATED(PointGravityObj%next)) THEN
       PointGravityObj%next%previous=>PointGravityObj%previous
    ELSE
       LastPointGravityObj=>PointGravityObj%previous
!       NULLIFY(LastPointGravityObj%next)
    END IF
    
  END SUBROUTINE RemovePointGravityObjFromList


!> Returns the number of point gravity source term objects
INTEGER FUNCTION PointGravity_CountObjects()

    TYPE(PointGravityDef), POINTER :: gravity_object

    INTEGER :: counter

    gravity_object => FirstPointGravityObj

    counter = 0

    DO WHILE(ASSOCIATED(gravity_object))
        counter = counter + 1
        gravity_object => gravity_object%next
    END DO

    PointGravity_CountObjects = counter

END FUNCTION PointGravity_CountObjects

!> Returns a pointer to the gravity object indicated by the ID, or a null pointer if the ID is invalid.
!! @param id An integer value (the ID of the point gravity object being sought)
SUBROUTINE FindPointGravityObject(id, gravity_object)

    INTEGER :: id
    TYPE(PointGravityDef), POINTER :: gravity_object

    TYPE(PointGravityDef), POINTER :: iter


    ! gravity_object will be returned as null if id is not found in the gravity object list.
    NULLIFY(gravity_object)

    iter => FirstPointGravityObj
    ! Loop over the gravity object list.  If an object with an ID matching id is found, then 
    ! associate the input pointer and cease iterating.
    DO WHILE (ASSOCIATED(iter))
        IF (iter%id == id) THEN
            gravity_object => iter
            EXIT
        END IF

        iter => iter%next

    END DO
END SUBROUTINE FindPointGravityObject

!> Initializes the datasets required to store point gravity source objects in a Chombo file.
!! @param chandle A chombo file handle.
!! @param obj_count The number of objects stour
SUBROUTINE PointGravity_InitChomboDatasets(chandle, obj_count)

    USE ChomboDeclarations, ONLY: ChomboHandle
    USE HDF5Declarations, ONLY:  Initialize_HDF5_Dataset_Int, Initialize_HDF5_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle

    INTEGER :: i_err
    INTEGER :: obj_count
    INTEGER :: iFixed


    IF (.NOT. ASSOCIATED(chandle)) THEN
        PRINT *, "PointGravity_InitChomboDatasets error::invalid Chombo handle."
        STOP
    END IF

    IF (obj_count < 0) THEN
        PRINT *, "PointGravity_InitChomboDatasets error::invalid object count ", obj_count, "."
        STOP
    END IF

    ! The size of the datasets is up to the user; they do not all need to be of length obj_count.
    CALL Initialize_HDF5_Dataset_Int("id", chandle%source_group_id, obj_count)

    ! Position coordinates
    CALL Initialize_HDF5_Dataset_Double("pos_x0", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("pos_y0", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("pos_z0", chandle%source_group_id, obj_count)

    ! Velocity coordinates
    CALL Initialize_HDF5_Dataset_Double("vel_x0", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("vel_y0", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("vel_z0", chandle%source_group_id, obj_count)

    CALL Initialize_HDF5_Dataset_Double("t0", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("mass", chandle%source_group_id, obj_count)
    CALL Initialize_HDF5_Dataset_Double("soft_length", chandle%source_group_id, obj_count)

    CALL Initialize_HDF5_Dataset_Int("soft_function", chandle%source_group_id, obj_count)

END SUBROUTINE PointGravity_InitChomboDatasets

!> Writes the data for a single point gravity source object to a Chombo file.
!! @param chandle A chombo file handle.
!! @param gravity_object A pointer to a point gravity source object.
SUBROUTINE PointGravity_WriteObjectToChombo(chandle, gravity_object)

    USE ChomboDeclarations, ONLY:  ChomboHandle
    USE HDF5Declarations, ONLY:  Write_Slab_To_Dataset_Int, Write_Slab_To_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle
    TYPE(PointGravityDef), POINTER :: gravity_object


    ! Write point gravity position variables.
    CALL Write_Slab_To_Dataset_Double("pos_x0", &
                                      chandle%source_group_id, &
                                      gravity_object%x0(1:1), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("pos_y0", &
                                      chandle%source_group_id, &
                                      gravity_object%x0(2:2), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("pos_z0", &
                                      chandle%source_group_id, &
                                      gravity_object%x0(3:3), &
                                      chandle%source_offset)

    ! Write point gravity velocity variables.
    CALL Write_Slab_To_Dataset_Double("vel_x0", &
                                      chandle%source_group_id, &
                                      gravity_object%v0(1:1), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("vel_y0", &
                                      chandle%source_group_id, &
                                      gravity_object%v0(2:2), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("vel_z0", &
                                      chandle%source_group_id, &
                                      gravity_object%v0(3:3), &
                                      chandle%source_offset)

    ! Point gravity time variable.
    CALL Write_Slab_To_Dataset_Double("t0", &
                                      chandle%source_group_id, &
                                      (/ gravity_object%t0 /), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("mass", &
                                      chandle%source_group_id, &
                                      (/ gravity_object%mass /), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Double("soft_length", &
                                      chandle%source_group_id, &
                                      (/ gravity_object%soft_length /), &
                                      chandle%source_offset)

    CALL Write_Slab_To_Dataset_Int("soft_function", &
                                   chandle%source_group_id, &
                                   (/ gravity_object%soft_function /), &
                                   chandle%source_offset)

    CALL Write_Slab_To_Dataset_Int("id", &
                                   chandle%source_group_id, &
                                   (/ gravity_object%id /), &
                                   chandle%source_offset)

    chandle%source_offset = chandle%source_offset + 1

END SUBROUTINE PointGravity_WriteObjectToChombo

!> Read the data for a single point gravity object in from a Chombo file.
!! @param chandle An active Chombo file handle object.
!! @param gravity_object An allocated PointGravityDef object.
SUBROUTINE PointGravity_ReadObjectFromChombo(chandle, gravity_object)

    USE ChomboDeclarations, ONLY: ChomboHandle
    USE HDF5Declarations, ONLY: Read_Slab_From_Dataset_Int, Read_Slab_From_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle
	TYPE(PointGravityDef), POINTER :: gravity_object

    INTEGER, DIMENSION(1), TARGET :: int_buffer_array
    REAL(KIND=qPrec), DIMENSION(1), TARGET :: dbl_buffer_array
    INTEGER, DIMENSION(:), POINTER :: int_buffer
    REAL(KIND=qPrec), DIMENSION(:), POINTER :: dbl_buffer


    int_buffer => int_buffer_array
    dbl_buffer => dbl_buffer_array

    int_buffer = 0
    dbl_buffer = 0.d0

    ! Read point gravity position variables.
    CALL Read_Slab_From_Dataset_Double("pos_x0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%x0(1) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("pos_y0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%x0(2) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("pos_z0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%x0(3) = dbl_buffer(1)


    ! Read point gravity velocity variables.
    CALL Read_Slab_From_Dataset_Double("vel_x0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%v0(1) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("vel_y0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%v0(2) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("vel_z0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%v0(3) = dbl_buffer(1)


    ! Point gravity time variable.
    CALL Read_Slab_From_Dataset_Double("t0", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%t0 = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("mass", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%mass = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("soft_length", &
                                      chandle%source_group_id, &
                                      dbl_buffer, &
                                      chandle%source_offset)

    gravity_object%soft_length = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Int("soft_function", &
                                   chandle%source_group_id, &
                                   int_buffer, &
                                   chandle%source_offset)

    gravity_object%soft_function = int_buffer(1)

    CALL Read_Slab_From_Dataset_Int("id", &
                                   chandle%source_group_id, &
                                   int_buffer, &
                                   chandle%source_offset)

    gravity_object%id = int_buffer(1)

    chandle%source_offset = chandle%source_offset + 1

END SUBROUTINE PointGravity_ReadObjectFromChombo

END MODULE PointGravitySrc
