!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    source_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir source
!! @brief directory containing modules for handling source terms

!> @defgroup Source Source Terms
!! @brief Group containing modules for handling source terms

!> @file source_control.f90
!! @brief Main file for module SourceControl

!> @defgroup SourceControl Source Control
!! @brief Module for managing various source terms
!! @ingroup Source

!> Module for managing various source terms
!! @ingroup SourceControl
MODULE SourceControl
  
  USE DataDeclarations
  USE PhysicsDeclarations
  USE SourceDeclarations
  USE EOS
  USE CoolingSrc
  USE CylindricalSrc
  USE UniformGravitySrc
  USE PointGravitySrc
  USE SelfGravitySrc
  USE RotatingSrc
!  USE OutflowSrc

  IMPLICIT NONE
  PRIVATE
  PUBLIC Src, SrcInit, SrcFinalizeInit, SrcInitTracers, SrcInitElliptics, ReadSourceObjects, SourceCell !CreateSrc, CoolingDef, CreateCoolingObject
  SAVE
  TYPE(SourcesDef),POINTER :: sources
  TYPE(LevelDef) :: currlevel

  ! sources verbosity shorthand
  INTEGER :: vb=0

  !> Generic interface for creating sources
  INTERFACE CreateSrc
	  MODULE PROCEDURE   CreateCoolingObject, CreatePointGravityObject!, CreateCylindricalObject, CreateOutflowObject &
                      !, CreateUniformGravityObject
  END INTERFACE
  

CONTAINS

  !> Read in and initialize parameters for source terms
  SUBROUTINE SrcInit(isrcsolvetype,iverbosity)
    INTEGER,OPTIONAL :: isrcsolvetype,iverbosity
    ! allocate sources & zero out components
    CALL CreateSourcesObj(sources,isrcsolvetype,iverbosity)
    CALL PointGravityInit()
    IF (lRestart)  CALL ReadSourceObjects(restart_frame)

  END SUBROUTINE SrcInit

  !> Finalize source term(s) initialization
  !! @details All source terms should be set up by user at this point, and
  !!            all hyperbolic, elliptic, etc., variables defined. Now,
  !!            set up any additional tables, etc.
  !!            [BDS][20110106]: Do not initialize cooling source objects on restarts--use the I/O routines for that.
  SUBROUTINE SrcFinalizeInit

    sources%lCooling=CoolingCheck()
    IF(sources%lCooling) CALL CoolingFinalizeInit

  END SUBROUTINE SrcFinalizeInit

  !> Calculate source terms and update q.
  !! @details Take in an Info structure, range, and hydro time step to 
  !!             integrate source term(s).
  !! @param Info Info structure
  !! @param mb Restricted range to calculate source on grid
  !! @param hdt Timestep from the hydro calculation
  SUBROUTINE Src(Info, mb, tnow, hdt)
    TYPE(InfoDef) :: Info
    INTEGER,INTENT(IN) :: mb(3,2)
    REAL(KIND=qPrec),INTENT(IN) :: hdt, tnow
    !
    REAL(KIND=xPrec) :: mx(3,2)
    LOGICAL :: lsrc
    IF(vb>0) PRINT*,'::Src begin'
    IF(vb>1) PRINT*,'      sources%iSrcSolveType',sources%iSrcSolveType

    IF(vb>0) THEN 
       PRINT*,'hydro dt is',hdt
       PRINT*,'extents are ',mb
    END IF

    currlevel=levels(Info%level)
    !mx(:,1)=()/currlevel%dx
    !mx(:,2)=()/currlevel%dx

    CALL Protectq(Info,mb, 'before source')

    SELECT CASE(sources%iSrcSolveType)
    CASE(ExplicitSource)
       CALL ExplicitSrc(Info, mb, tnow, hdt)
    CASE(ImplicitSource)
       CALL ImplicitSrc(Info, mb, tnow, hdt)
    CASE(ExactSource)
       CALL ExactSrc(Info, mb, tnow, hdt)
    CASE(NoSource)
    CASE DEFAULT
       PRINT*,'source_control.f90::Src error -- Unknown source solve type requested. Halting.'
       STOP
    END SELECT
    CALL Protectq(Info,mb, 'after source')
    IF(vb>0) PRINT*,'::Src complete'
  END SUBROUTINE Src


  ! ==================================================================
  ! =                   Explicit Scheme Section                      =
  ! ==================================================================


  !> Use subcycling algorithm to calculate source terms.
  !! @details When there are no strong source terms present,
  !!               algorithm uses 2nd order RK to quickly update q. 
  !!               When stiff source terms present, it switches to 
  !!               4th/5th order RK.
  !! @param Info Info structure
  !! @param mb Restricted range to calculate source on grid
  !! @param hdt Timestep from the hydro calculation
  SUBROUTINE ExplicitSrc(Info, mb, tnow, hdt)
    ! Interface declarations
    TYPE(InfoDef) :: Info
    INTEGER,INTENT(IN) :: mb(3,2)
    REAL(KIND=qPrec),INTENT(IN) :: hdt
    ! Internal declarations
    REAL(KIND=qPrec) :: pos(3)
    REAL(KIND=qPrec) :: dq(NrHydroVars),tnow,tnext, dv
    INTEGER :: lvl,i,j,k,ip(3)
    REAL(KIND=qPrec) :: error, volumefactor
    LOGICAL :: success, ghost, ghostz, ghostyz
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q
    dv = Levels(Info%level)%dx**ndim
    ALLOCATE(q(NrHydroVars))
    DO k=mb(3,1),mb(3,2)
       Ghostz = .NOT. (k >= 1 .AND. k <= Info%mx(3))
       DO j=mb(2,1),mb(2,2)
          Ghostyz = Ghostz .OR. .NOT. (j >= 1 .AND. j <= Info%mx(2))
          DO i=mb(1,1),mb(1,2)
             Ghost = Ghostyz .OR. .NOT. (i >= 1 .AND. i <= Info%mx(1))
             IF (Ghost) THEN 
                VolumeFactor = 0d0
             ELSE 
                IF (Info%Level==MaxLevel) THEN
                   VolumeFactor = 1d0
                ELSE 
                   IF (Info%ChildMask(i,j,k) >= 1) THEN
                      VolumeFactor = 0d0
                   ELSE
                      VolumeFactor = 1d0
                   END IF
                END IF
             END IF
  
             ! Method: 
             !   1) Try 2nd order Runge-Kutta to get estimate of error for timestep.
             !      Should speed things up where source terms are weak.
             !   2) If error is above tolerance, reduce timestep and switch to
             !       5th/4th order RK to do the integration.
             !   3) Iterate at 5th/4th until full timestep reached.

             q=Info%q(i,j,k,1:NrHydroVars)
             CALL Cons_to_prim(q)
             ip=(/i,j,k/)
             


             CALL SourceCell(q, Info, ip, tnow, hdt, dv*VolumeFactor, PRIMITIVE)!where it gets complicated
             CALL Prim_to_cons(q)
             IF(ANY(q(:).ne.q(:) .OR. abs(q(:)).gt.huge(abs(q(:)))) .AND. .NOT. lRequestRestart) THEN
                PRINT*, 'Src routine produced a NAN'
                CALL PrintQ(Info, q, tnow, i, j, k)
                lRequestRestart=.true.
                !          STOP
                DEALLOCATE(q)
                RETURN          
             END IF
             Info%q(i,j,k,1:NrHydroVars)=q

          END DO
       END DO
    END DO
    DEALLOCATE(q)
    !IF(vb>0) THEN

    !END IF
  END SUBROUTINE ExplicitSrc

  SUBROUTINE SourceCell(q, Info, ip, tnow, hdt, dv, lForm)
     ! Interface declarations
     TYPE(InfoDef) :: Info
     REAL(KIND=qPrec),INTENT(IN) :: hdt
     ! Internal declarations
     REAL(KIND=qPrec) :: q(:)
     REAL(KIND=qPrec) :: pos(3)
     REAL(KIND=qPrec) :: dq(NrHydroVars),tnow,tnext, dv
     INTEGER :: ip(3)
     REAL(KIND=qPrec) :: error
     LOGICAL :: success
     LOGICAL :: lForm

     tnext=tnow+hdt
     pos(1:nDim)=Info%xBounds(1:nDim,1)+(REAL(ip(1:nDim))-half)*levels(Info%level)%dx
     IF (nDim == 2) pos(3)=Info%xBounds(3,1)
     IF (iCylindrical == NOCYL) CALL PointGravity(q,hdt,pos,tnow,dv,info%Level,lform)
!     CALL PointGravity(q,hdt,pos,tnow,dv,info%Level,lform)
     ! 2nd order RK, uses hdt to get full timestep error
     CALL RKOrder2(q,dq,pos,tnow,hdt,error,Info,ip,lform)
     IF(vb>1) PRINT*,'::ExplicitSrc:RKOrder2 -- dq,error',dq,error

     IF(Error > sources%SrcTol) THEN
        IF(vb>1) PRINT*,'::ExplicitSrc:RKOrder45, qbefore, tnow, tnext',q,tnow,tnext
        ! iterate with 5th/4th order RK to get error & adjust timestep
        ! until full timestep is completed
        ! The subroutine updates q
        CALL RKOrder45(q,pos,hdt,tnow,tnext,Info,ip,success,lform)
        IF (.NOT. success .AND. .NOT. lRequestRestart) THEN             
           write(*,*) 'RK Failed'
           CALL PrintQ(Info, q, tnow, ip(1), ip(2), ip(3))
           lRequestRestart=.true.
           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k-1,iPhi))
           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k,iPhi))
           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k+1,iPhi))

           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k-1,iPhiDot))
           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k,iPhiDot))
           !             CALL OutputDoubleArray(Info%q(i-1:i+1,j-1:j+1,k+1,iPhiDot))
           !             STOP
           RETURN
        END IF
        IF(vb>1) PRINT*,'::ExplicitSrc:RKOrder45, qafter, tnow, tnext',q,tnow,tnext

     ELSE
        ! good enough at 2nd order, update q
        q=q+dq
     END IF

  END SUBROUTINE SourceCell



  !> 2nd order Runge-Kutta scheme
  !! @params q  Fluid variables for this cell
  !! @params dq Change in q from source terms
  !! @params pos  Location of cell
  !! @params dt Timestep to integrate over
  SUBROUTINE RKOrder2(q,dq,pos,t,dt,error,Info,ip,lform)
     ! Interface declarations
     TYPE(InfoDef) :: Info
     INTEGER :: ip(3)
     REAL(KIND=qPrec) :: q(:)
     REAL(KIND=qPrec) :: dq(NrHydroVars), pos(3), dt, error,t
     ! Internal declarations
     REAL(KIND=qPrec), ALLOCATABLE :: qstar(:)
     REAL(KIND=qPrec) :: k1(NrHydroVars),k2(NrHydroVars), qerr(NrHydroVars), minscales(NrHydroVars)
     LOGICAL :: lform
     dq=0d0

     ALLOCATE(qStar(NrHydroVars))
     ! Runge-Kutta 2nd order: for a function y(t), going from t=t0:t1 with dt=t1-t0,
     !   the second order integration is
     !         k1 = d(y0)/dt
     !         y* = y0 + dt k1
     !         k2 = d(y*)/dt
     !         y1 = y0 + dt/2 (k1+k2)
     !   where in our case y(t) -> q(t) and dq/dt is calculated from the source terms.
     !   The error is
     !        error = |y*/y1|

     k1    = SrcDerivs(q,pos,t,Info,ip, lform)
     WHERE(k1/=k1)k1=0d0
     qstar = q + dt*k1
     k2    = SrcDerivs(qstar,pos,t+dt,Info,ip, lform)
     WHERE(k2/=k2)k2=0d0

     dq=5d-1*dt*(k1+k2)

     IF(ALL(ABS(dq(:))<TINY(dq(:)))) THEN
        dq=0d0
        error=0d0
     ELSE
        minscales(1)=minDensity
        minscales(m_low:m_high)=max(max(minDensity, q(1))*Iso_Speed,sqrt(SUM(q(m_low:m_high)**2)))
        IF (iE .ne. 0) minscales(iE)=max(minDensity, q(1))*Iso_Speed2
        IF (lMHD) minscales(iBx:iBz) = max(max(minDensity, q(1))*Iso_Speed2,sqrt(SUM(q(iBx:iBz)**2)))
        IF (nTracerLo /= 0) minscales(nTracerLo:nTracerHi) = minDensity          
        qerr=q+dq-qstar
        error=MAXVAL(ABS(qErr(:))/max(abs(q(:)), minscales(:)))
        !       error = MAXVAL(ABS(qstar(:))/ABS(q(:)+dq(:))) !5d-1*dt*(k1+k2)))
     END IF





     DEALLOCATE(qStar)
  END SUBROUTINE RKOrder2


  !> Combination 5th/4th order Runge-Kutta scheme
  !! @details Algorithm adjusts timestep to subcycle until full timestep is integrated over.
  !!            Method is based on the Cash-Karp RK algorithm in Numerical Recipes (1992), 710ff.
  !!            It takes advantage of the fact that six function calls results in formulas
  !!            for both 4th and 5th order integration, allowing quick ability to measure error
  !!            and adjust timestep as appropriate. 
  !!         Once the error is acceptable, it uses the 5th order result to update q and continue.
  !! @params q  Fluid variables for this cell
  !! @params pos  Location of cell
  !! @params hdt Timestep to integrate over
  !! @params htnow Time at beginning of timestep
  !! @params htnext Time at end of timestep
  SUBROUTINE RKOrder45(q,pos,hdt,htnow,htnext,Info,ip,success,lform)
     ! Interface declarations
     TYPE(InfoDef) :: Info
     INTEGER :: ip(3)
     REAL(KIND=qPrec) :: q(:)
     REAL(KIND=qPrec) :: pos(3),hdt,htnow,htnext
     ! Internal declarations
     REAL(KIND=qPrec) :: dt,tnow,tnext
     INTEGER,PARAMETER :: MaxIters=1d4
     INTEGER :: niter
     REAL(KIND=qPrec),ALLOCATABLE :: qStar(:)
     ! CKRK stuff, see above reference for table
     REAL(KIND=qPrec) :: k1(NrHydroVars),k2(NrHydroVars),k3(NrHydroVars),k4(NrHydroVars),k5(NrHydroVars),k6(NrHydroVars), qTemp(NrHydroVars), qErr(NrHydroVars), maxerror, minscales(NrHydroVars)
     REAL(KIND=qPrec),PARAMETER ::  A1=0.0,           A2=0.2,        A3=0.3,          A4=0.6,             A5=1.0, A6=0.875 &
          , B21=0.2                                                                                 &
          , B31=3./40.,       B32=9./40.                                                            &
          , B41=0.3,          B42=-0.9,      B43=1.2                                                &
          , B51=-11./54.,     B52=2.5,       B53=-70./27.,    B54=35./27.                           &
          , B61=1631./55296., B62=175./512., B63=575./13824., B64=44275./110592., B65=253./4096.    &
          ,  C1=37./378.,         C3=250./621.,         C4=125./594.,                           C6=512./1771. &
          , DC1=C1-2825./27648., DC3=C3-18575./48384., DC4=C4-13525./55296., DC5=-277./14336., DC6=C6-0.25
     REAL(KIND=qPrec),PARAMETER :: SAFETY=0.9, PSHRNK=-0.25, PGROW=-0.2
     LOGICAL :: success
     LOGICAL :: lform
     !
     ALLOCATE(qStar(NrHydroVars))
     dt=hdt
     tnow=htnow
     tnext=htnext

     k1=0d0;k2=0d0;k3=0d0;k4=0d0;k5=0d0;k6=0d0

     DO niter=1,MaxIters
        !
        ! Cash-Karp RK functions, see above reference for details
        !
        k1 = SrcDerivs(q,pos,tnow,Info,ip,lform)
        WHERE(k1/=k1)k1=0
        qStar = q + dt*( B21*k1                                     )
        k2 = SrcDerivs(qStar,pos,tnow+dt*A2,Info,ip,lform)
        WHERE(k2/=k2)k2=0
        qStar = q + dt*( B31*k1 + B32*k2                            )
        k3 = SrcDerivs(qStar,pos,tnow+dt*(A3),Info,ip,lform)
        WHERE(k3/=k3)k3=0
        qStar = q + dt*( B41*k1 + B42*k2 + B43*k3                   )
        k4 = SrcDerivs(qStar,pos,tnow+dt*(A4),Info,ip,lform)
        WHERE(k4/=k4)k4=0
        qStar = q + dt*( B51*k1 + B52*k2 + B53*k3 + B54*k4          )
        k5 = SrcDerivs(qStar,pos,tnow+dt*(A5),Info,ip,lform)
        WHERE(k5/=k5)k5=0
        qStar = q + dt*( B61*k1 + B62*k2 + B63*k3 + B64*k4 + B65*k5 )
        k6 = SrcDerivs(qStar,pos,tnow+dt*(A6),Info,ip,lform)
        WHERE(k6/=k6)k6=0
        qTemp = q + dt*(  C1*k1 +  C3*k3 +  C4*k4          +  C6*k6 )
        qErr  =     dt*( DC1*k1 + DC3*k3 + DC4*k4 + DC5*k5 + DC6*k6 )

        ! 
        ! Error evaluation & timestep adjustment
        !
!        IF(vb>1) WRITE(*,'(A,10E20.12)')'qtemp ',qtemp(1)
!        IF(vb>1) WRITE(*,'(A,10E20.12)')'qerr  ',qerr(1)
        IF(vb>1) PRINT*,'qtemp ',qtemp
        IF(vb>1) PRINT*,'qerr  ',qerr       
        minscales(1)=minDensity
        minscales(m_low:m_high)=max(minDensity, q(1))*Iso_Speed
        IF (iE .ne. 0) minscales(iE)=max(minDensity, q(1))*Iso_Speed2
        IF (lMHD) minscales(iBx:iBz) = max(minDensity, q(1))*Iso_Speed2
        IF (nTracerLo /= 0) minscales(nTracerLo:nTracerHi) = minDensity          
        maxerror=MAXVAL(ABS(qErr(:))/max(abs(q(:)), minscales(:)))
        IF (vb>1) print*,maxerror
        IF(maxerror > sources%SrcTol) THEN
           ! reduce timestep and retry
           dt = MAX( SAFETY*dt*(maxerror/sources%SrcTol)**PSHRNK, dt/MaxIters)
           IF(dt <= TINY(dt) .AND. .NOT. lRequestRestart) THEN
              PRINT*,'Src Error::RKOrder45 -- timestep insignificant. Halting.'
              !             PRINT*,'  Iterations/MaxIters: ',niter, MaxIters
              !             PRINT*,' Last timestep, full timestep: ',dt, hdt
              !             PRINT*,' Current time, hydro begin/end time: ',tnow,htnow,htnext
              !             PRINT*,' Time left to go: ',htnow-tnow
              CALL PrintQ(Info, q, htnow, ip(1),ip(2),ip(3))
              lRequestRestart=.true.
              success=.false.
              EXIT
              !             STOP
              !             RETURN

           END IF
           ! tnow remains the same; update tnext
           tnext=tnow+dt
        ELSE
           q = qTemp
           tnow=tnow+dt
           dt = MAX( MIN( SAFETY*dt*(maxerror/sources%SrcTol)**PGROW, 5*dt, htnext-tnow), 0d0)
           tnext=tnow+dt
           IF(dt <= TINY(dt)) THEN
              ! end of full timestep reached
              success=.true.
              EXIT
           END IF
        END IF
     END DO

     IF(niter>MaxIters .AND. .NOT. lRequestRestart) THEN 
        PRINT*,'Src Error::RKOrder45 -- maximum number of iterations reached.'
        !      PRINT*,'  Max iterations: ',MaxIters
        !      PRINT*,' Last timestep, full timestep: ',dt, hdt
        !      PRINT*,' Current time, hydro begin/end time: ',tnow,htnow,htnext
        !      PRINT*,' Time left to go: ',htnow-tnow
        CALL PrintQ(Info, q, htnow, ip(1), ip(2), ip(3))
        !       STOP
        lRequestRestart=.true.
        success=.false.
     END IF
     DEALLOCATE(qStar)
  END SUBROUTINE RKOrder45



  ! ==================================================================
  ! =                   Implicit Scheme Section                      =
  ! ==================================================================

  !> Use implicit algorithm for calculating source terms.
  !! @details In certain cases, the equations for the source terms
  !!          may be cast in an implicit formalism. This requires
  !!          some manner of iterative scheme to converge.
  !! @param Info Info structure
  !! @param mb Restricted range to calculate source on grid
  !! @param hdt Timestep from the hydro calculation
  SUBROUTINE ImplicitSrc(Info, mb, tnow,hdt)
     TYPE(InfoDef) :: Info
     INTEGER,INTENT(IN) :: mb(3,2)
     REAL(KIND=qPrec),INTENT(IN) :: hdt,tnow
     IF(vb>0) PRINT*,'::ImplicitSrc begin'
     IF(vb>0) PRINT*,'::ImplicitSrc complete'
  END SUBROUTINE ImplicitSrc


  ! ==================================================================
  ! =                   Exact Scheme Section                         =
  ! ==================================================================

  !> Use the exact scheme (for cooling only?) of Townsend, 2009.
  !! @details For simple (piecewise-power-law) cooling, the 
  !!               evolution may be integrated exactly.
  !! @param Info Info structure
  !! @param mb Restricted range to calculate source on grid
  !! @param hdt Timestep from the hydro calculation
  SUBROUTINE ExactSrc(Info, mb, tnow,hdt)
     TYPE(InfoDef) :: Info
     INTEGER,INTENT(IN) :: mb(3,2)
     REAL(KIND=qPrec),INTENT(IN) :: tnow,hdt
     IF(vb>0) PRINT*,'::ExactSrc begin'
     IF(vb>0) PRINT*,'::ExactSrc complete'
  END SUBROUTINE ExactSrc



  ! ==================================================================
  ! =            ~~~ Wrapper Subroutines/Functions ~~~               =
  ! ==================================================================

  !> Changes to q (dqdt) for all source terms
  !! @params q Fluid variables for this cell
  !! @params pos  Location of cell
  FUNCTION SrcDerivs(q,pos,t,Info,ip,lform)
     ! Interface declarations
     TYPE(InfoDef) :: Info
     INTEGER :: ip(3)
     REAL(KIND=qPrec) :: q(:)
     REAL(KIND=qPrec) :: pos(3), SrcDerivs(NrHydroVars), t
     ! Internal declarations
     REAL(KIND=qPrec) :: dqdt(NrHydroVars)
     LOGICAL :: lform
     dqdt=0d0

     !IF(sources%lCooling)        
     CALL        Cooling(q,dqdt,pos,currlevel%dx,lform)

     IF (iCylindrical.ge.NoCyl   .and.   iCylindrical.le.WithAngMom) THEN
        IF (iCylindrical.ne.NoCyl) CALL Cylindrical(q,dqdt,pos,lform)!,currlevel%dx)
     ELSE
        print*,'';print*,'iCylindrical= ',iCylindrical
        print*,'but should be either 0, 1 or 2. Aborting.'
        stop
     END IF

     IF(luniformgravity) CALL UniformGravity_src(q,dqdt,t,lform)
     IF (OmegaRot /= 0d0) CALL Rotating(q, dqdt, pos, lform)
     IF (iCylindrical /= NOCYL) CALL PointGravity_inst(q,dqdt,pos,t,lform) !moved to source cell for improved mom. cons.
!     CALL PointGravity_inst(q,dqdt,pos,t,lform) !moved to source cell for improved mom. cons.

     !    IF(sources%lOutflows)       CALL       Outflows(q,dqdt,pos)

!     IF (lSelfGravity .AND. SourceMethod == OPERATORSPLIT) CALL   SelfGravity(q,dqdt,pos,currlevel%dx,t,Info,ip,lform)
     !Self Gravity now always implemented in hyperbolic solve directly
     SrcDerivs=dqdt
  END FUNCTION SrcDerivs


  !> Initialize indices and fields for source tracers
  !! @params NrVars number of variables
  !! @params NrTracerVars number of tracer fields
  SUBROUTINE SrcInitTracers
     ! Interface declarations
     !IF(sources%lCooling)        
     CALL     InitCoolingTracers(sources)
     !IF(sources%lCylindrical)    CALL InitCylindricalTracers(NrVars,NrTracerVars,sources)
     !    IF(sources%lUniformGravity) CALL InitUniformGravTracers(NrVars,NrTracerVars,sources)
     !     CALL   InitPointGravTracers(NrVars,NrTracerVars,sources)
     !    IF(sources%lOutflows)       CALL     InitOutflowTracers(NrVars,NrTracerVars,sources)
  END SUBROUTINE SrcInitTracers

  !> Initialize indices and fields for source tracers
  !! @params NrVars number of variables
  !! @params NrEllipticVars number of elliptic fields
  SUBROUTINE SrcInitElliptics()
     ! Interface declarations
     !IF(sources%lCooling)        
     CALL     InitCoolingElliptics(sources)
     !IF(sources%lCylindrical)    CALL InitCylindricalElliptics(NrVars,NrEllipticVars,sources)
     !    IF(sources%lUniformGravity) CALL InitUniformGravElliptics(NrVars,NrEllipticVars,sources)
     !    IF(sources%lPointGravity)   CALL   InitPointGravElliptics(NrVars,NrEllipticVars,sources)
     !    IF(sources%lOutflows)       CALL     InitOutflowElliptics(NrVars,NrEllipticVars,sources)
  END SUBROUTINE SrcInitElliptics


  !> Reads source-term objects from a Chombo file.  Currently, only cooling is implemented.
  !! @param nframe The number of the restart frame.
  SUBROUTINE ReadSourceObjects(nframe)

     USE ChomboDeclarations, ONLY: ChomboHandle, CreateChomboHandle, CloseChomboHandle, CHOMBO_HANDLE_READ, &
          Chombo_OpenSourceGroup, Chombo_CloseSourceGroup

     USE CoolingSrc, ONLY: CoolingDef, CreateCoolingObject, Cooling_ReadObjectFromChombo

     USE PointGravitySrc, ONLY: PointGravityDef, CreatePointGravityObject, PointGravity_ReadObjectFromChombo

     INTEGER :: nframe

     ! Variable declarations
     TYPE(ChomboHandle), POINTER :: chandle
     TYPE(CoolingDef), POINTER :: coolingobj
     TYPE(PointGravityDef), POINTER :: gravity_obj
     INTEGER :: i_err
     INTEGER :: nr_objects
     CHARACTER(LEN=23) :: s_filename


     ! Open a reading handle for the specified frame.
     WRITE(s_filename, '(A10,I5.5,A4)') 'out/chombo', nframe, '.hdf'
     CALL CreateChomboHandle(s_filename, chandle, CHOMBO_HANDLE_READ)

!!! Check For Cooling Terms !!!

     ! Open the 'cooling_objects' group and save the handle.
     nr_objects = Chombo_OpenSourceGroup(chandle, "cooling_objects")

     chandle%source_offset = 0

     ! Read source objects from the data file until the expected number of objects is read.
     DO WHILE (chandle%source_offset < nr_objects)

        ! Create a new cooling object.
        NULLIFY(coolingobj)
        CALL CreateCoolingObject(coolingobj)

        ! Read in the cooling data from the Chombo file.  This subroutine also advances the
        ! chandle%source_offset variable.
        CALL Cooling_ReadObjectFromChombo(chandle, coolingobj)

     END DO

     ! Close the source term group.  This also clears the source term offset.
     CALL Chombo_CloseSourceGroup(chandle)

!!! End Check for Cooling Terms !!!

!!! Begin Check For Point Gravity Terms !!!
     ! Open the 'point_gravity_objects' group and save the handle.
     nr_objects = Chombo_OpenSourceGroup(chandle, "point_gravity_objects")

     chandle%source_offset = 0

     ! Read source objects from the data file until the expected number of objects is read.
     DO WHILE (chandle%source_offset < nr_objects)

        ! Create a new gravity object.
        NULLIFY(gravity_obj)
        CALL CreatePointGravityObject(gravity_obj)

        ! Read in the point gravity data from the Chombo file.  This subroutine also advances the
        ! chandle%source_offset variable.
        CALL PointGravity_ReadObjectFromChombo(chandle, gravity_obj)

     END DO

     ! Close the source term group.  This also clears the source term offset.
     CALL Chombo_CloseSourceGroup(chandle)




     CALL CloseChomboHandle(chandle)

  END SUBROUTINE ReadSourceObjects

  ! ==========================================
  ! =      Sources creation/destruction      =
  ! =      and list manipulation section     =
  ! ==========================================

  !> Create a new sources object and initialize it
  !! @params sources The sources object to be created in the sources module
  !! @params isrcsolvetype integer setting kind of source integration
  !! @params iverbosity integer setting verbosity level of source
  SUBROUTINE CreateSourcesObj(sources,isrcsolvetype,iverbosity)
     ! Interface declarations
     TYPE(SourcesDef),POINTER :: sources
     INTEGER,OPTIONAL :: isrcsolvetype,iverbosity

     IF(ASSOCIATED(sources)) THEN
        PRINT*,'source_control.f90::CreateSourcesObj error -- sources already associated. Halting.'
        STOP
     END IF

     IF(PRESENT(iverbosity)) THEN
        vb=iverbosity
     ELSE
        vb=0
     END IF

     ALLOCATE(sources)
     ! Initialize everything
     sources%iSrcSolveType=explicitSource
     sources%level=0
     sources%lPrimitive=.FALSE.
     sources%SrcTol=SrcPrecision     ! tolerance for source terms (may vary with AMR level)
     !sources%lCooling=.FALSE.
     !sources%lCylindrical=.FALSE.
     !sources%lUniformGravity=.FALSE.
     !sources%lPointGravity=.FALSE.
     !sources%lOutflows=.FALSE.
     !
     IF(PRESENT(isrcsolvetype)) sources%iSrcSolveType=isrcsolvetype
     sources%iverbosity=vb

     IF(vb>0) PRINT*,'::CreateSourcesObj successful'
  END SUBROUTINE CreateSourcesObj

  !> Empty subroutine at present, anticipating only ever needing one sources object
  SUBROUTINE DestroySourcesObj
  END SUBROUTINE DestroySourcesObj



END MODULE SourceControl

