//#########################################################################
//		
//    Copyright (C) 2003-2012 Department of Physics and Astronomy,
//                            University of Rochester,
//                            Rochester, NY
//
//    fpthread_lib.c is part of AstroBEAR.
//
//    AstroBEAR is free software: you can redistribute it and/or modify	  
//    it under the terms of the GNU General Public License as published by 
//    the Free Software Foundation, either version 3 of the License, or    
//    (at your option) any later version.
//
//    AstroBEAR is distributed in the hope that it will be useful, 
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
//
//#########################################################################
/*! \file fpthread_lib.c
\brief Wrapper file for pthreads
*/
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>


//////////////////////////////////////////////
/// Thread creation and destruction
//////////////////////////////////////////////
  

void fpthread_create_(pthread_t *thread, pthread_attr_t *attr, void (*start_routine)(void*),void *data,int *ierr)
{
  
  *ierr = pthread_create(thread,attr, (void*)start_routine,data);  
   return;
}

void fpthread_create_noattr_(pthread_t *thread, void (*start_routine)(void*),void *data,int *ierr)
{
  *ierr = pthread_create(thread,NULL,(void*)start_routine,data);  
   return;
}

void fpthread_join_(pthread_t *thread, void **status, int *ierr)
{
  *ierr = pthread_join(*thread,(void**) status);
  return;
}

void fpthread_cancel_(pthread_t *thread,int*ierr)
{
   *ierr = pthread_cancel(*thread); 
   return;
}

void fpthread_detach_(pthread_t *thread, int *ierr)
{
  *ierr = pthread_detach(*thread);
}

void fpthread_exit_(void *data)
{
  pthread_exit((void *)data);
}

void fpthread_setschedprio_(pthread_t *thread, int *prio, int *ierr)
{
  *ierr= pthread_setschedprio(*thread, *prio);
}	

void fpthread_getschedparam_(pthread_t *thread, int *policy, int *priority, int *ierr)
{
  struct sched_param param;
  *ierr=pthread_getschedparam(*thread, policy, &param);
 *priority=param.sched_priority;
}


void fpthread_setschedparam_(pthread_t *thread, int *policy, int* priority, int *ierr)
{
  struct sched_param param;
  param.sched_priority=*priority;
  *ierr=pthread_setschedparam(*thread, *policy, &param);
}
//////////////////////////////////////////////////
/// bookkeeping support
////////////////////////////////////////////////// 
void fpthread_equal_(pthread_t *p1, pthread_t *p2,int *iequal,int *ierr)
{
   *iequal=pthread_equal(*p1,*p2); 
   *ierr = 0;
}

void fpthread_self_(pthread_t *pout,int *ierr)
{
  *pout = pthread_self();
  *ierr = 0;
}

void get_errno_(int *e)
{
	*e = errno;
}


//////////////////////////////////////////////////
/// MUTEX support
////////////////////////////////////////////////// 
void fpthread_mutex_init_(pthread_mutex_t *mutex, pthread_mutexattr_t *attr, int *ierr)
{ 
   *ierr = pthread_mutex_init(mutex,attr);
   return;
}

void fpthread_mutex_lock_(pthread_mutex_t*mutex,int *ierr)
{
   *ierr = pthread_mutex_lock(mutex);
}

void fpthread_mutex_trylock_(pthread_mutex_t*mutex,int *ierr)
{
   *ierr = pthread_mutex_trylock(mutex);
}

void fpthread_mutex_unlock_(pthread_mutex_t*mutex,int *ierr)
{
   *ierr = pthread_mutex_unlock(mutex);
}

void fpthread_mutex_destroy_(pthread_mutex_t*mutex,int *ierr)
{
   *ierr = pthread_mutex_destroy(mutex);
}

//////////////////////////////////////////
/// CONDITION VARIABLE support
/// not yet supported: pthread_cond_timedwait
//////////////////////////////////////////

void fpthread_cond_init_(pthread_cond_t *cond, pthread_condattr_t *attr, int *ierr)
{ 
   *ierr = pthread_cond_init(cond,attr);
}

void fpthread_cond_signal_(pthread_cond_t *cond,int *ierr)
{ 
   *ierr = pthread_cond_signal(cond);
}

void fpthread_cond_broadcast_(pthread_cond_t *cond,int *ierr)
{ 
   *ierr = pthread_cond_broadcast(cond);
}

void fpthread_cond_wait_(pthread_cond_t *cond,pthread_mutex_t*mutex,int *ierr)
{ 
   *ierr = pthread_cond_wait(cond,mutex);
}

void fpthread_cond_destroy_(pthread_cond_t *cond,int *ierr)
{ 
   *ierr = pthread_cond_destroy(cond);
}


//////////////////////////////////////////
/// ATTRIBUTE support 
//////////////////////////////////////////

void fpthread_attr_destroy_(pthread_attr_t *attr, int *ierr)
{
  *ierr = pthread_attr_destroy(attr);
}

void fpthread_attr_init_(pthread_attr_t *attr, int *ierr)
{
  *ierr = pthread_attr_init(attr);
}

void fpthread_attr_getdetachstate_(pthread_attr_t *attr, int *detachstate, int *ierr)
{
   *ierr = pthread_attr_getdetachstate(attr, detachstate);
}

void fpthread_attr_setdetachstate_(pthread_attr_t *attr, int *detachstate, int *ierr)
{
  //  printf("%d  %d \n",PTHREAD_SCOPE_SYSTEM, PTHREAD_SCOPE_PROCESS);
  *ierr = pthread_attr_setdetachstate(attr, *detachstate);
}

void fpthread_attr_getstacksize_(pthread_attr_t *attr, size_t *stacksize, int *ierr)
{
   *ierr = pthread_attr_getstacksize(attr, stacksize);
}

void fpthread_attr_setstacksize_(pthread_attr_t *attr, size_t *stacksize, int *ierr)
{
  *ierr = pthread_attr_setstacksize(attr, *stacksize);
}

void fpthread_attr_getstack_(pthread_attr_t *attr, size_t *stackaddr, size_t *stacksize, int *ierr)
{
  void * cstackaddr;
  *ierr = pthread_attr_getstack(attr, &cstackaddr, stacksize);
  *stackaddr = (int *)cstackaddr;
}

void fpthread_attr_getscope_(pthread_attr_t *attr, int *scope, int *ierr)
{
   *ierr = pthread_attr_getscope(attr, scope);
}

void fpthread_attr_setscope_(pthread_attr_t *attr, int *scope, int *ierr)
{
  *ierr = pthread_attr_setscope(attr, *scope);
}

void fpthread_attr_getschedpolicy_(pthread_attr_t *attr, int *schedpolicy, int *ierr)
{
   *ierr = pthread_attr_getschedpolicy(attr, schedpolicy);
}

void fpthread_attr_setschedpolicy_(pthread_attr_t *attr, int *schedpolicy, int *ierr)
{
  *ierr = pthread_attr_setschedpolicy(attr, *schedpolicy);
}

void fpthread_attr_getschedparam_(pthread_attr_t *attr, struct sched_param *schedparam, int *ierr)
{
   *ierr = pthread_attr_getschedparam(attr, schedparam);
}

void fpthread_attr_setschedparam_(pthread_attr_t *attr, struct sched_param *schedparam, int *ierr)
{
  *ierr = pthread_attr_setschedparam(attr, schedparam);
}



//make functions void and tack on return value at end
//duplicate call passing in components just with their names
//Any non-pointer input arguments need to be made to pointers in both the declaration and the pthread call
