!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    tree_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file tree_declarations.f90
!! @brief Main file for module TreeDeclarations

!> @defgroup TreeDeclarations Tree Declarations
!! @brief Module defining tree node structure and basic operations
!! @ingroup TreeOps

!> Module defining tree node structure and basic operations
!! @ingroup TreeDeclarations
!================================================================================
! Module Name:  TreeDeclarations
! Module File:  tree_declarations.f90
! Purpose:      Describe the node structure and the methods that operate upon it.
! Created:      20100625 by Brandon Shroyer.
! Modified:     20100628 Overlap calculation methods added by Jonathan Carroll.
!================================================================================
MODULE TreeDeclarations
   USE DataDeclarations
   USE GlobalDeclarations

   IMPLICIT NONE
   PRIVATE

   ! Public Data types
   PUBLIC :: NodeBox, NodeBoxList, NodeDef, NodeDefList, pNodeDef, pNodeDefList

   !Routines for creating nodes
   PUBLIC :: AddNode, AddOldNode, AddFindNode, AddFindOldNode

   !Routines for destroying nodes
   PUBLIC :: NullifyNodeFields, DestroyNode, BackupNode

   !Routines for adding connections to tree
   PUBLIC :: AddParent, AddOverlap, AddFindOverlap, AddNeighbor, AddChild, AddFindChild

   !Routines for managing nodelists
   PUBLIC :: AddNodeToList, ClearNodeList, DestroyNodeList, BackupNodeList

   !Routines for finding nodes
   PUBLIC :: FindNode, StrictFindNode, FindOldNode, StrictFindOldNode, GetChildID, FindNodeInList, FindBackupNode, StrictFindBackupNode, FindParent, StrictFindParent, StrictFindAnyNode

   !Miscellaneous Routines
   PUBLIC :: isAncestor, GetFinestLevel, NodeCount, traverse, CellCount

   !Routines for determining necessary connections in the tree
   PUBLIC :: ChildCanBeNeighbor, NephewCanBeNeighbor, NephewCanBeOverlap, Overlaps, Neighbors

   !Routines for manipulating node box indices
   PUBLIC :: GetChildMGlobal, GetChildMBounds

   !Routines for manipulating node boxes
   PUBLIC :: CreateNodeBox, DestroyNodeBox, MatchBox

   !Routines for manipulating nodeboxlists
   PUBLIC :: AddNodeBoxToList, ClearNodeBoxList, NodeBoxCount

   SAVE

   !> Contains the processor ID and grid dimensions of the data the node represents.
   TYPE NodeBox
      SEQUENCE
      INTEGER :: mGlobal(3,2)
      INTEGER :: MPI_ID
   End TYPE NodeBox

   !> List of NodeBoxes
   TYPE NodeBoxList
      SEQUENCE
      TYPE(NodeBox) :: self
      TYPE(NodeBoxList), POINTER :: next
   End TYPE NodeBoxList

   !> Contains information needed to manage nodes
   TYPE NodeDef
      SEQUENCE
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: parent
      TYPE(NodeDefList), POINTER :: children
      TYPE(NodeDefList), POINTER :: oldchildren
      TYPE(NodeDefList), POINTER :: neighbors
      TYPE(NodeDefList), POINTER :: overlaps
      TYPE(NodeDefList), POINTER :: lastchild
      TYPE(NodeDefList), POINTER :: lastneighbor
      TYPE(NodeDefList), POINTER :: lastoverlap
      REAL, POINTER :: ProcTime(:)
      INTEGER, POINTER :: ProcList(:)
      TYPE(InfoDef), POINTER :: Info
      REAL(KIND=qPrec) :: stamp
      INTEGER :: ID
   End TYPE NodeDef

   !> List of NodeDefs
   TYPE NodeDefList
      SEQUENCE
      TYPE(NodeDef), POINTER :: self
      TYPE(NodeDefList), POINTER :: next
   End TYPE NodeDefList

   !> Contains a Pointer to a NodeDefList
   TYPE pNodeDefList
      SEQUENCE
      TYPE(NodeDefList), POINTER :: p
   End TYPE pNodeDefList

   !> Contains a Pointer to a NodeDef
   TYPE pNodeDef
      SEQUENCE
      TYPE(NodeDef), POINTER :: p
   End TYPE pNodeDef

   TYPE(pNodeDefList), DIMENSION(:), POINTER, PUBLIC :: Nodes, OldNodes, ExternalNodes, OldExternalNodes, BackupNodes, BackupExternalNodes
   TYPE(pNodeDefList), DIMENSION(:), POINTER, PUBLIC :: LastLocalNode, LastExternalNode, LastOldLocalNode, LastOldExternalNode

   INTEGER :: nNodes=0
   !=========
CONTAINS 
   !=========

   !> @name Routines for creating nodes
   !! @{

   !> Create new node structure and add it to the appropriate current nodelist (Nodes(level)%p).
   !! @param level level of new node
   !! @param box new node's box
   !! @param node pointer to added node
   !! @param proclist new node's processor list (optional)
   !! @param proctime new node's processor times (optional)
   SUBROUTINE AddNode(level, box, node, proclist, proctime)

      INTEGER :: level
      TYPE(NodeBox) :: box
      INTEGER, OPTIONAL, DIMENSION(:), POINTER :: proclist
      REAL, OPTIONAL, DIMENSION(:), POINTER :: proctime
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: LastNode, nodelist
      INTEGER :: iErr

      IF (ASSOCIATED(node)) THEN
         PRINT *, "node already associated!!!"
         STOP
      END IF

      !      PRINT "('AddNode(', i2, ') = [', 6i4, ']')", level, box%mGlobal
      ! Nodes where the data is going to reside on another processor should go
      ! in the external nodes list.
      IF (box%MPI_ID == MPI_ID) THEN
         CALL AddNodeToList(node,LastLocalNode(level)%p, Nodes(level)%p)
      ELSE
         CALL AddNodeToList(node,LastExternalNode(level)%p, ExternalNodes(level)%p)
      END IF
      node%box=box
      !  Assign a processor pool and timeslot to the node if they are supplied.
      IF (PRESENT(proclist)) THEN
         node%proclist=>proclist
         IF (PRESENT(proctime)) THEN
            node%proctime=>proctime
         ELSE
            IF (ASSOCIATED(node%proctime)) THEN
               PRINT*, "node%proctime already associated"
               STOP
            END IF
            NULLIFY(node%proctime)
            ALLOCATE(node%proctime(SIZE(proclist)), STAT=iErr)
            IF (iErr /= 0) THEN
               PRINT *, "AddNode error: unable to allocate proctime list."
               STOP
            END IF
            node%proctime = 0.0
         END IF
      END IF

      ! If box data is to reside on this processor, then create a InfoDef structure.

      ! Let's do this later after we've built the tree so we can pass in parents? jjc
      !    IF (box%MPI_ID == MPI_ID) THEN
      !        ALLOCATE(node%Info)
      !        CALL InitInfo(node%info)
      !    END IF
      nNodes=nNodes+1
      node%iD=nNodes
!      IF (box%MPI_id == MPI_id) THEN
!         PRINT "('***AddNode(', i2, '), [', 6i4, ']) success = ', l2, ' ID=', i4,i4,i4' .')", level, &
!              box%mGlobal, ASSOCIATED(Nodes(level)%p), node%id, node%box%MPI_ID
!      ELSE
!         PRINT "('***AddExternalNode(', i2, '), [', 6i4, ']) success = ', l2, ' ID=', i4,i4,i4' .')", level, &
!              box%mGlobal, ASSOCIATED(ExternalNodes(level)%p), node%id, node%box%MPI_ID
!      END IF
      CALL CPU_TIME(node%stamp)
   END SUBROUTINE AddNode


   !> Create new node structure and add it to the appropriate old nodelist (OldNodes(level)%p).
   !! @param level level of new node
   !! @param box new node's box
   !! @param node pointer to added node
   !! @param proclist new node's processor list (optional)
   !! @param proctime new node's processor times (optional)
   SUBROUTINE AddOldNode(level, box, node, proclist, proctime)

      INTEGER :: level
      TYPE(NodeBox) :: box
      INTEGER, OPTIONAL, DIMENSION(:), POINTER :: proclist
      REAL, OPTIONAL, DIMENSION(:), POINTER :: proctime
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: LastNode, nodelist
      INTEGER :: iErr

      IF (ASSOCIATED(node)) THEN
         PRINT *, "node already associated!!!"
         STOP
      END IF

      !      PRINT "('AddNode(', i2, ') = [', 6i4, ']')", level, box%mGlobal
      ! Nodes where the data is going to reside on another processor should go
      ! in the external nodes list.
      IF (box%MPI_ID == MPI_ID) THEN
         CALL AddNodeToList(node,LastOldLocalNode(level)%p, OldNodes(level)%p)
         !         LastNode=>LastLocalNode(level)%p
      ELSE
         CALL AddNodeToList(node,LastOldExternalNode(level)%p, OldExternalNodes(level)%p)
         !         LastNode=>LastExternalNode(level)%p
      END IF
      node%box=box
      !  Assign a processor pool and timeslot to the node if they are supplied.
      IF (PRESENT(proclist)) THEN
         node%proclist=>proclist
         IF (PRESENT(proctime)) THEN
            node%proctime=>proctime
         ELSE
            IF (ASSOCIATED(node%proctime)) THEN
               PRINT*, "node%proctime already associated"
               STOP
            END IF
            NULLIFY(node%proctime)
            ALLOCATE(node%proctime(SIZE(proclist)), STAT=iErr)
            IF (iErr /= 0) THEN
               PRINT *, "AddNode error: unable to allocate proctime list."
               STOP
            END IF
            node%proctime = 0.0
         END IF
      END IF

      ! If box data is to reside on this processor, then create a InfoDef structure.

      ! Let's do this later after we've built the tree so we can pass in parents? jjc
      !    IF (box%MPI_ID == MPI_ID) THEN
      !        ALLOCATE(node%Info)
      !        CALL InitInfo(node%info)
      !    END IF
      nNodes=nNodes+1
      node%iD=nNodes
!            IF (box%MPI_id == MPI_id) THEN
!               PRINT "('***AddOldNode(', i2, '), [', 6i4, ']) success = ', l2, ' ID=', i4, i4, ' .')", level, &
!                      box%mGlobal, ASSOCIATED(OldNodes(level)%p), node%id, node%box%mpi_id
!            ELSE
!               PRINT "('***AddOldExternalNode(', i2, '), [', 6i4, ']) success = ', l2, ' ID=', i4,i4, ' .')", level, &
!                      box%mGlobal, ASSOCIATED(OldExternalNodes(level)%p), node%id, node%box%mpi_id
!            END IF

   END SUBROUTINE AddOldNode



   !> Finds or Adds a node structure in the appropriate node list (Nodes(level)%p).
   !! @param level level of node
   !! @param box node's box
   !! @param node pointer to added/found node
   !! @param proclist node's processor list (optional)
   !! @param proctime node's processor times (optional)

   SUBROUTINE AddFindNode(level, box, node, proclist, proctime)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node
      !TYPE(NodeDef), POINTER, INTENT(OUT) :: node
      INTEGER, OPTIONAL, DIMENSION(:), POINTER :: proclist
      REAL, OPTIONAL, DIMENSION(:), POINTER :: proctime


      CALL FindNode(level, box, node)

      IF (ASSOCIATED(node)) THEN
         ! If node is found and there is a processor pool/time list supplied,
         ! then assign the optional lists.

         IF (present(proclist)) THEN
            node%proclist=>proclist
            IF (present(proctime))  node%proctime=>proctime
         END IF
      ELSE
         ! If node is not found, then create it.
         IF(present(proclist)) THEN
            IF (present(proctime)) THEN
               CALL AddNode(level,box,node,proclist,proctime)
            ELSE
               CALL AddNode(level,box,node,proclist)
            END IF
         ELSE
            CALL AddNode(level,box,node)
         END IF
      END IF

   END SUBROUTINE AddFindNode

   !> Finds or Adds a node structure in the appropriate node list (OldNodes(level)%p).
   !! @param level level of node
   !! @param box node's box
   !! @param node pointer to added/found node
   !! @param proclist node's processor list (optional)
   !! @param proctime node's processor times (optional)
   SUBROUTINE AddFindOldNode(level, box, node, proclist, proctime)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER, INTENT(OUT) :: node
      INTEGER, OPTIONAL, DIMENSION(:), POINTER :: proclist
      REAL, OPTIONAL, DIMENSION(:), POINTER :: proctime


      CALL FindOldNode(level, box, node)

      IF (ASSOCIATED(node)) THEN
         ! If node is found and there is a processor pool/time list supplied,
         ! then assign the optional lists.
         IF (present(proclist)) THEN
            node%proclist=>proclist
            IF (present(proctime))  node%proctime=>proctime
         END IF
      ELSE
         ! If node is not found, then create it.
         IF(present(proclist)) THEN
            IF (present(proctime)) THEN
               CALL AddOldNode(level,box,node,proclist,proctime)
            ELSE
               CALL AddOldNode(level,box,node,proclist)
            END IF
         ELSE
            CALL AddOldNode(level,box,node)
         END IF
      END IF

   END SUBROUTINE AddFindOldNode

   !> @}


   !> @name Node destruction routines
   !! @{

   !> Nullifes fields associated with a node
   !! @param node node object
   SUBROUTINE NullifyNodeFields(node)

      TYPE(NodeDef), POINTER :: node

      NULLIFY(node%proclist)
      NULLIFY(node%proctime)
      NULLIFY(node%parent)
      NULLIFY(node%Info)
      NULLIFY(node%children, node%lastchild)
      NULLIFY(node%oldchildren)
      NULLIFY(node%neighbors, node%lastneighbor)
      NULLIFY(node%overlaps, node%lastoverlap)

   END SUBROUTINE NullifyNodeFields


   !> Deallocates structures associated with a node
   !! @param node node object
   SUBROUTINE DestroyNode(node)

      TYPE(NodeDef), POINTER :: node
      TYPE(InfoDef), POINTER :: Info
      
      IF (ASSOCIATED(node%children))  CALL ClearNodeList(node%children)
      IF (ASSOCIATED(node%oldchildren)) CALL ClearNodeList(node%oldchildren)
      !    CALL ClearNodeList(node%parent)
      IF (ASSOCIATED(node%overlaps, target=node%neighbors)) THEN
         NULLIFY(node%neighbors)
      ELSE
         CALL ClearNodeList(node%neighbors)
      END IF
      CALL ClearNodeList(node%overlaps)
      NULLIFY(node%lastchild)
      NULLIFY(node%lastneighbor)
      NULLIFY(node%lastoverlap)
      NULLIFY(node%parent)

      IF (ASSOCIATED(node%Info)) CALL DestroyInfo(node%Info)     
      IF (ASSOCIATED(node%ProcList)) THEN
         DEALLOCATE(node%ProcList)
         NULLIFY(node%ProcList)
      END IF
      IF (ASSOCIATED(node%proctime)) THEN
         DEALLOCATE(node%proctime)
         NULLIFY(node%proctime)
      END IF
      DEALLOCATE(node)
      NULLIFY(node)

   END SUBROUTINE DestroyNode


   !> Routine that duplicates a node
   !! @param original original node
   !! @param node backup node
   SUBROUTINE BackupNode(original, node, lRestore)
      TYPE(NodeDef), POINTER :: original, node
      LOGICAL :: lRestore
      node%box=original%box
      node%stamp=original%stamp
      node%id=original%id
      IF (ASSOCIATED(original%proctime)) THEN
         ALLOCATE(node%proctime(size(original%proctime)))
         node%proctime=original%proctime
      END IF
      IF (ASSOCIATED(original%proclist)) THEN
         ALLOCATE(node%proclist(size(original%proclist)))
         node%proclist=original%proclist
      END IF
      IF (ASSOCIATED(original%info)) THEN
         CALL BackupInfo(original%info, node%info, lRestore)
      END IF                 
   END SUBROUTINE BackupNode


   !> @}




   !> @name Routines for adding connections in tree
   !! @{

   !> Assigns the parent of a node
   !! @param node current node
   !! @param parent parent node
   SUBROUTINE AddParent(node, parent)

      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: parent

!          PRINT "('***AddParent():  node ID = ', I4, '; parent ID = ', I4, '.')", node%ID, parent%ID
      node%Parent=>parent
   END SUBROUTINE AddParent


   !> Singular version of AddOverlaps(), adds one new node to a node's overlaps list.
   !! @param node current node
   !! @param overlap overlap node
   SUBROUTINE AddOverlap(node,overlap)

      TYPE(NodeDef), POINTER :: node, overlap
      INTEGER :: iErr
!      PRINT "('***AddOverlap:   Node: ', 2I4, '[', 6I4, '], Overlap: ', 2I4, ', [', 6I4, ']')", node%box%mpi_ID, node%id, node%box%mGlobal, overlap%box%mpi_ID, overlap%id, overlap%box%mGlobal
      CALL AddNodeToList(overlap, node%lastoverlap, node%overlaps)

   END SUBROUTINE AddOverlap

   !> Find or append an overlap to a node's overlap list
   !! @param node current node
   !! @param overlap The overlap node being found or added.
   SUBROUTINE AddFindOverlap(node, overlap)
      TYPE(NodeDefList), POINTER :: overlaplist
      TYPE(NodeDef), POINTER :: node, overlap, temp

      overlaplist=>node%overlaps

      ! Iterate over list to find a box that matches the target box dimensions.
      DO WHILE (associated(overlaplist))
         temp=>overlaplist%self
         IF (MatchBox(temp%box,overlap%box)) EXIT
         overlaplist=>overlaplist%next
      END DO

      IF (.NOT. ASSOCIATED(overlaplist))  CALL AddOverlap(node,overlap)

   END SUBROUTINE AddFindOverlap


   !> Append a single new node to a node's neighbors list.
   !! @param node current node
   !! @param neighbor neighbor node
   SUBROUTINE AddNeighbor(node,neighbor)

      TYPE(NodeDef), POINTER :: node, neighbor
      INTEGER :: iErr
!      PRINT "('***AddNeighbor:  Node: ', 2I4, '[', 6I4, '], Neighbor: ', 2I4, ', [', 6I4, ']')", node%box%MPI_ID, node%ID, node%box%mGlobal, neighbor%box%MPI_ID, neighbor%ID, neighbor%box%mGlobal
      CALL AddNodeToList(neighbor, node%lastneighbor, node%neighbors)
   END SUBROUTINE AddNeighbor


   !> Append new entries to a node's child list.
   !! @param node current node
   !! @param child child nodes
   SUBROUTINE AddChild(node, child)
      TYPE(NodeDef), POINTER :: node, child
      INTEGER :: iErr
!         PRINT "('***AddChild:  Node: ', I4, '[', 6I4, '], Child: ', I4, ', [', 6I4, ']')", node%ID, node%box%mGlobal, child%ID, child%box%mGlobal
      CALL AddNodeToList(child, node%lastchild, node%children)

   END SUBROUTINE AddChild

   !> Find or append a child to a node's child lsit
   !! @param node current node
   !! @param child child to find or add
   SUBROUTINE AddFindChild(node, child)
      TYPE(NodeDefList), POINTER :: childlist
      TYPE(NodeDef), POINTER :: node, child, temp

      childlist=>node%children

      ! Iterate over list to find a box that matches the target box dimensions.
      DO WHILE (associated(childlist))
         temp=>childlist%self
         IF (MatchBox(temp%box,child%box)) EXIT
         childlist=>childlist%next
      END DO

      IF (.NOT. Associated(childlist))  CALL AddChild(node,child)

   END SUBROUTINE AddFindChild

   !> @}


   !> @name NodeList Operations
   !! @{

   !> Adds node to nodelist and updates lastnode pointer
   !! @param node node object to add
   !! @param lastnode pointer to last node of nodelist
   !! @param nodelist nodelist to add to
   SUBROUTINE AddNodeToList(node, lastnode, nodelist)
      TYPE(NodeDefList), POINTER :: lastnode
      TYPE(NodeDefList), POINTER, OPTIONAL :: nodelist
      TYPE(NodeDef), POINTER :: node
      INTEGER :: iErr
      integer :: ierr2

      IF (.NOT. ASSOCIATED(LastNode)) THEN     
         ALLOCATE(LastNode, STAT=iErr)
         NULLIFY(LastNode%next)
         NULLIFY(LastNode%self)
         IF (iErr /= 0) THEN
            PRINT *, "AddNode() error: unable to allocate LastNode list object."
            STOP
         END IF
         IF (present(nodelist)) nodelist=>lastnode
      ELSE
         IF (ASSOCIATED(LastNode%next)) THEN
            PRINT *, "Error - last node next allocated"
            STOP
         END IF
         ALLOCATE(LastNode%next, STAT=iErr)
         IF (iErr /= 0) THEN
            PRINT *, "AddNode() error: unable to allocate LastNode%next list object."
            STOP
         END IF
         LastNode=>LastNode%next         
         NULLIFY(LastNode%next)
         NULLIFY(LastNode%self)

      END IF

      IF (ASSOCIATED(node)) THEN
         LastNode%self=>node
      ELSE
         ALLOCATE(LastNode%self, STAT=iErr)

         IF (iErr /= 0) THEN
            PRINT *, "AddNode() error: unable to allocate LastNode%self object."
            STOP
         END IF
         node=>LastNode%self
         NULLIFY(node%parent, node%children, node%oldchildren, node%neighbors, node%overlaps, node%lastchild, &
              node%lastneighbor, node%lastoverlap, node%proctime, node%proclist, node%info)         
      END IF

   END SUBROUTINE AddNodeToList


   !> Deallocate all the node objects in a given node list.
   !! @param nodelist nodelist to clear
   RECURSIVE SUBROUTINE ClearNodeList(nodelist)
      TYPE(NodeDefList), POINTER :: nodelist
      IF (.NOT. ASSOCIATED(nodelist))  RETURN
      IF (ASSOCIATED(nodelist%next)) CALL ClearNodeList(nodelist%next)
      DEALLOCATE(nodelist)
      NULLIFY(nodelist)
   END SUBROUTINE ClearNodeList

   !> Deallocates and nullifies all links in a NodeList list.  Unlike ClearNodeList(), 
   !! @details DestroyNodeList() also destroys the Node object associated with each list entry.
   !! @param nodelist The pointer to a NodeList object
   RECURSIVE SUBROUTINE DestroyNodeList(nodelist)

      TYPE(NodeDefList), POINTER :: nodelist
      IF (.NOT. ASSOCIATED(nodelist))  RETURN
      IF (ASSOCIATED(nodelist%next))  CALL DestroyNodeList(nodelist%next)
      IF (ASSOCIATED(nodelist%self))  CALL DestroyNode(nodelist%self)
      DEALLOCATE(nodelist)
      NULLIFY(nodelist)

   END SUBROUTINE  DestroyNodeList

   !> Routine that duplicates a nodelist
   !! @param original original nodelist
   !! @param backup backup nodelist
   SUBROUTINE BackupNodelist(original, backup, lRestore, lastnode)
      TYPE(NodeDefList), POINTER :: original, backup, lastbackup, nodelist
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER, OPTIONAL :: lastnode
      LOGICAL :: lRestore
      !      IF (ASSOCIATED(original)) ALLOCATE(backup)
      NULLIFY(lastbackup, backup)
      nodelist=>original
      DO WHILE (ASSOCIATED(nodelist))
         ALLOCATE(node)
         CALL NullifyNodeFields(node)
         CALL BackupNode(nodelist%self, node, lRestore)
         CALL AddNodeToList(node, lastbackup, backup)
         nodelist=>nodelist%next
      END DO
      IF (PRESENT(LastNode)) LastNode=>lastbackup

   END SUBROUTINE BackupNodelist


   !> @}

   !> @name Routines for finding nodes
   !! @{


   !>  Find a node in the appropriate node list (Nodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   SUBROUTINE FindNode(level,box,node)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node

      IF (box%MPI_ID==MPI_ID) THEN
         CALL FindNodeInList(box,Nodes(level)%p, node)
      ELSE
         CALL FindNodeInList(box,ExternalNodes(level)%p, node)
      END IF

   END SUBROUTINE FindNode

   !>  Find a node in the appropriate node list or STOP! (Nodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   !! @param caller string passed in by calling routine
   SUBROUTINE StrictFindNode(level, box, node, caller)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node
      CHARACTER(*) :: caller


      CALL FindNode(level, box, node)

      IF (.NOT. ASSOCIATED(node)) THEN
         PRINT *, caller, "::StrictFindNode() error: node not associated."
         PRINT "('StrictFindNode(box%MPI_id = ', i4, ', level = ', i2, ') failed on box [', 6I5, ']')", box%MPI_id, level, box%mGlobal
         STOP
      END IF

   END SUBROUTINE StrictFindNode


   !>  Find a node in the appropriate node list or STOP - using only mGlobal and not mpi_id
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   !! @param caller string passed in by calling routine

   SUBROUTINE StrictFindAnyNode(level, box, node, caller)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node
      CHARACTER(*) :: caller
      TYPE(NodeDefList), POINTER :: nodelist
      NULLIFY(node)
      nodelist=>Nodes(level)%p
      DO WHILE (ASSOCIATED(nodelist))
         IF (ALL(nodelist%self%box%mGlobal==box%mGlobal)) THEN
            node=>nodelist%self
            EXIT
         END IF
         nodelist=>nodelist%next
      END DO
      IF (.NOT. ASSOCIATED(node)) THEN
         
         nodelist=>ExternalNodes(level)%p
         DO WHILE (ASSOCIATED(nodelist))
            IF (ALL(nodelist%self%box%mGlobal==box%mGlobal)) THEN
               node=>nodelist%self
               EXIT
            END IF
            nodelist=>nodelist%next
         END DO
      END IF

      IF (.NOT. ASSOCIATED(node)) THEN
         PRINT *, caller, "::StrictFindAnyNode() error: node not associated."
         PRINT "('StrictFindAnyNode(box%MPI_id = ', i4, ', level = ', i2, ') failed on box [', 6I5, ']')", box%MPI_id, level, box%mGlobal
         STOP
      END IF
   END SUBROUTINE StrictFindAnyNode


   !>  Find a node in the appropriate node list or STOP! (Nodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   !! @param caller string passed in by calling routine
   SUBROUTINE StrictFindBackupNode(level, box, node, caller)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node
      CHARACTER(*) :: caller


      CALL FindBackupNode(level, box, node)

      IF (.NOT. ASSOCIATED(node)) THEN
         PRINT *, caller, "::StrictFindBackupNode() error: node not associated."
         PRINT "('StrictFindBackupNode(box%MPI_id = ', i4, ', level = ', i2, ') failed on box [', 6I5, '] on processor ', i5)", box%MPI_id, level, box%mGlobal, MPI_ID
         STOP
      END IF

   END SUBROUTINE StrictFindBackupNode

   !>  Find a node in the appropriate node list (OldNodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   SUBROUTINE FindOldNode(level,box,node)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node

      ! Nodes where the data is going to reside on another processor should go
      ! in the external nodes list.
      IF (box%MPI_ID==MPI_ID) THEN
         CALL FindNodeInList(box,OldNodes(level)%p, node)
      ELSE
         CALL FindNodeInList(box,OldExternalNodes(level)%p, node)
      END IF
   END SUBROUTINE FindOldNode


   !>  Find a node in the appropriate node list (OldNodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   SUBROUTINE FindBackupNode(level,box,node)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node

      ! Nodes where the data is going to reside on another processor should go
      ! in the external nodes list.
      IF (box%MPI_ID==MPI_ID) THEN
         CALL FindNodeInList(box,BackupNodes(level)%p, node)
      ELSE
         CALL FindNodeInList(box,BackupExternalNodes(level)%p, node)
      END IF
   END SUBROUTINE FindBackupNode



   SUBROUTINE FindNodeInList(box,nodelist,node)
      TYPE(Nodebox) :: box
      TYPE(NodeDef), POINTER :: node, temp
      TYPE(NodeDefList), POINTER :: nodelist, traverse

      ! Make sure that the pointer is null by default.
      NULLIFY(node)
      traverse=>nodelist
      ! Iterate over list to find a box that matches the target box dimensions.
      DO WHILE (associated(traverse))
         temp=>traverse%self

         IF (MatchBox(temp%box,box)) THEN
            node=>temp
            EXIT
         END IF

         traverse=>traverse%next
      END DO

   END SUBROUTINE FindNodeInList

   !> Find a node in the appropriate node list or STOP! (OldNodes(level)%p)
   !! @param level level to search
   !! @param box box to match
   !! @param node pointer to found node
   !! @param caller string passed in by calling routine
   SUBROUTINE StrictFindOldNode(level, box, node, caller)

      INTEGER :: level
      TYPE(NodeBox) :: box
      TYPE(NodeDef), POINTER :: node
      CHARACTER(*) :: caller

      CALL FindOldNode(level, box, node)

      IF (.NOT. ASSOCIATED(node)) THEN
         PRINT *, caller, "::StrictFindOldNode() error: node not associated."
         PRINT "('StrictFindNode(box%MPI_id = ', i4, ') failed on box [', 6I5, ']')", box%MPI_id, box%mGlobal
         STOP
      END IF

   END SUBROUTINE StrictFindOldNode

   !> Find the position of a child within a node's childlist
   !! @param node node object
   !! @param child child to find
   INTEGER FUNCTION GetChildID(node,child)
      TYPE(NodeDefList), POINTER :: childlist
      TYPE(NodeDef), POINTER :: node, child
      INTEGER :: i
      i=0
      childlist=>node%children
      DO WHILE (ASSOCIATED(childlist))
         i=i+1
         IF (MatchBox(child%box,childlist%self%box)) THEN
            GetChildID=i
            RETURN
         END IF
         childlist=>childlist%next
      END DO
      PRINT*, "Amr_node.f90: Couldn't find child! Stopping..."
      STOP
   END FUNCTION GetChildID

   !> @}


   !> @name Miscellaneous routines
   !! @{

   !> Checks a childmask to determine whether or not it corresponds to an ancestor cell
   !! @param childmask value of childmask
   LOGICAL FUNCTION isAncestor(childmask)
      INTEGER :: childmask
      isAncestor = (childmask < 0 .AND. childmask /= NEIGHBORCHILD)
   END FUNCTION isAncestor

   !> Finds the highest level ON THIS PROCESS that has any grids on it.
   !! On multiprocessor jobs, this is followed by an MPI_ALLREDUCE to aggregate the results.
   FUNCTION GetFinestLevel() RESULT(level_buf)
      INTEGER :: level
      INTEGER :: finest_level
      INTEGER :: level_buf
      INTEGER :: iErr


      finest_level = Maxlevel

      ! Loop over the levels and retrieve the finest level on this processor.
      DO level=0, MaxLevel
         IF (NodeCount(Nodes(level)%p) == 0) THEN
            finest_level = level-1 
            EXIT
         END IF
      END DO

      ! Gathers the FinestLevel values from all the processors, finds the maximum, and 
      ! redistributes the maximum to all processors.
      CALL MPI_ALLREDUCE(finest_level, level_buf, 1, MPI_INTEGER, MPI_MAX, MPI_COMM_WORLD, iErr)

   END FUNCTION GetFinestLevel

   !> Counts number of nodes in a nodelist
   !! @param node_list nodelist object
   INTEGER FUNCTION NodeCount(node_list)

      TYPE(NodeDefList), POINTER :: node_list, temp

      INTEGER :: iCount
      iCount=0

      temp=>node_list
      DO WHILE (ASSOCIATED(temp))
         IF (ASSOCIATED(temp%self))  iCount = iCount + 1
         temp => temp%next
      END DO

      NodeCount = iCount
   END FUNCTION NodeCount


   !> Counts number of nodes in a nodelist
   !! @param node_list nodelist object
   INTEGER FUNCTION CellCount(node_list)

      TYPE(NodeDefList), POINTER :: node_list, temp
      CellCount=0

      temp=>node_list
      DO WHILE (ASSOCIATED(temp))
        ! IF (ASSOCIATED(temp%self%box%MPI_ID == MPI_ID))  
         CellCount = CellCount + product(temp%self%box%mGlobal(:,2)-temp%self%box%mGlobal(:,1)+1)
         temp => temp%next
      END DO
   END FUNCTION CellCount

   !> Traverses a nodelist printing out useful information
   !! @param node_list nodelist object
   SUBROUTINE traverse(node_list)

      TYPE(NodeDefList), POINTER :: node_list, temp
      INTEGER :: ierr 
      write(*,*) "Traversing NodeList"
      temp=>node_list
      DO WHILE (ASSOCIATED(temp))
         IF (ASSOCIATED(temp%self)) THEN
            !            write(*,'(7I4)') temp%self%box%mGlobal, temp%self%id
            PRINT "('Node ID ', I5, '; box = ', 7I4, '.')", temp%self%id, temp%self%box%mGlobal,temp%self%box%mpi_id
            IF (.NOT. ASSOCIATED(temp%self%info)) THEN
               PRINT *, "WARNING:  No Info associated."
            END IF
         ELSE
            write(*,*) "node not associated"
            write(*,*) ierr
         END IF

         temp => temp%next
      END DO
   END SUBROUTINE traverse

   !> @}

   !> @name Routines for determining necessary connections in tree
   !! @{

   !> Determines whether or not a child needs to be communicated to neighboring nodes
   !! @param node node object
   !! @param child node's child
   !! @param level level
   LOGICAL FUNCTION ChildCanBeNeighbor(node,child,level)
      ! Neighbors need to know about children that will have common fluxes
      ! and/or data for setting ghost for the second step on the child level
      ! (ie within mbc cells).  Children that are not well within their parents
      ! can potentially neighbor neighbor's children.

      TYPE(NodeDef) :: node, child
      INTEGER, DIMENSION(3,2) :: mGlobal
      INTEGER :: level, mbc

      !    mbc=hyperbolic_mbc  !<-this works         !levels(level)%gmbc(1)
      mbc=levels(level+1)%nmbc

      mGlobal=LevelUp(node%box%mGlobal,level)
      !      mGlobal(1:nDim,2)=node%box%mGlobal(1:nDim,2)*2
      !      mGlobal(1:nDim,1)=(node%box%mGlobal(1:nDim,1)-1)*2

      IF (MaintainAuxArrays) THEN
         mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     + mbc + 1
         mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     - mbc - 1
      ELSE
         mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     + mbc
         mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     - mbc
      END IF

      ChildCanBeNeighbor = .NOT. Within(child%box%mGlobal,mGlobal)

   END FUNCTION ChildCanBeNeighbor


   !> Determines whether or not box1 is entirely within box 2
   !! @param mGlobal1 box 1
   !! @param mGlobal2 box 2
   LOGICAL FUNCTION Within(mGlobal1,mGlobal2)
      INTEGER, DIMENSION(3,2) :: mGlobal1, mGlobal2
      Within= ALL(mGlobal1(1:nDim,1) >= mGlobal2(1:nDim,1)) &
           .AND. ALL(mGlobal1(1:nDim,2) <= mGlobal2(1:nDim,2))
   END FUNCTION Within

   !> Determines whether or not a child could neighbor the children of a neighbor
   !! @param neighbor neighboring node
   !! @param child child node
   !! @param neighborlevel level of neighbor node
   LOGICAL FUNCTION NephewCanBeNeighbor(neighbor,child,neighborlevel)
      ! Neighbors need to know about children that will have common fluxes
      ! and/or data for setting ghost for the second step on the child level
      ! (ie within mbc cells).  This routine checks for children within mbc cells 
      ! of potential neighbor children.

      TYPE(NodeDef) :: neighbor, child
      INTEGER :: neighborlevel,i,j,k,mbc
      INTEGER, DIMENSION(3,2) :: mGlobal, ioffset


      !    mbc=hyperbolic_mbc   !<- This works   !levels(neighborlevel)%gmbc(1)
      mbc=levels(neighborlevel+1)%nmbc
      NephewCanBeNeighbor=.false.
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=LevelUp(neighbor%box%mGlobal(:,:)+SPREAD((/i,j,k/)*levels(neighborlevel)%mx(:),2,2),neighborlevel)

               IF (MaintainAuxArrays) THEN
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     - mbc - 1
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     + mbc + 1
               ELSE
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     - mbc
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     + mbc
               END IF
               NephewCanBeNeighbor =  NephewCanBeNeighbor .OR. BoxOverlap(mGlobal,child%box%mglobal)
            END DO
         END DO
      END DO
   END FUNCTION NephewCanBeNeighbor

   !> Determines whether or not a child could overlap the children of an overlap
   !! @param overlap overlapping node
   !! @param child child node
   !! @param level level of overlap node
   LOGICAL FUNCTION NephewCanBeOverlap(overlap,child,level)
      ! Overlaps need to know about children that will need to send or receive data
      ! for the extended ghost zones of new boxs.  This routine checks whether or not
      ! the given child can overlap within rmbc of the overlap's potential children.

      TYPE(NodeDef) :: overlap, child
      INTEGER, DIMENSION(3,2) :: mGlobal,ioffset
      INTEGER :: level,i,j,k,rmbc

      !    NephewCanBeOverlap = .TRUE.
      !    RETURN

      !    rmbc=levels(level)%gmbc(levels(level)%step)
      !    rmbc=levels(level+1)%gmbc(1) !<- This works
      rmbc=levels(level+1)%ombc(1)

      !      rmbc=mbc*levels(level)%CoarsenRatio
      NephewCanBeOverlap=.false.
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=LevelUp(overlap%box%mGlobal(:,:)+SPREAD((/i,j,k/)*levels(level)%mx(:),2,2),level)

               IF (MaintainAuxArrays) THEN
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     - rmbc - 1
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     + rmbc + 1
               ELSE
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1)     - rmbc
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2)     + rmbc
               END IF
               NephewCanBeOverlap = NephewCanBeOverlap .OR. BoxOverlap(child%box%mGlobal,mGlobal)
            END DO
         END DO
      END DO
   END FUNCTION NephewCanBeOverlap

   !> Determines whether or not a child could overlap the child of an overlap
   !! @param child child node
   !! @param overlapchild child of overlap node
   !! @param level level of child and overlapchild
   LOGICAL FUNCTION Overlaps(child,overlapchild,level)
      ! This routine checks whether or not the extended region of one box overlaps with the 
      ! non-extended region of the other.  Will determine whether or not children need to share
      ! data with eachother before or after their two steps.

      TYPE(NodeDef) :: child, overlapchild
      INTEGER :: rmbc,level,i,j,k

      INTEGER, DIMENSION(3,2) :: mGlobal,ioffset
      !    rmbc=levels(level)%gmbc(levels(level)%step)  !<- Not sure why this worked... Reduced size of ghosted region to not include extended zones......
      rmbc=levels(level)%ombc(1)
      Overlaps=.false.
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=overlapchild%box%mGlobal(:,:)+SPREAD((/i,j,k/)*levels(level)%mX(:),2,2)

               IF (MaintainAuxArrays) THEN
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1) - rmbc - 1
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2) + rmbc + 1
               ELSE
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1) - rmbc
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2) + rmbc
               END IF


               Overlaps=Overlaps .OR. BoxOverlap(child%box%mGlobal, mGlobal)
            END DO
         END DO
      END DO
   END FUNCTION Overlaps

   !> Determines whether or not a child could neighbor the child of a neighbor
   !! @param child child node
   !! @param neighborchild child of neighbor node
   !! @param level level of child and neighborchild
   LOGICAL FUNCTION Neighbors(child,neighborchild,level)
      ! This routine checks to see if two boxs are within mbc of each other.  This determins
      ! whether boxs will need to synchronize fluxes/emf's and share data in between the
      ! first and second step.

      TYPE(NodeDef) :: child, neighborchild

      INTEGER, DIMENSION(3,2) :: mGlobal,ioffset
      INTEGER :: i,j,k,level,mbc

      !    mbc=hyperbolic_mbc    !<- This worked     !levels(level)%gmbc(1)
      mbc=levels(level)%nmbc
      Neighbors=.false.
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=neighborchild%box%mGlobal(:,:)+SPREAD((/i,j,k/)*levels(level)%mX(:),2,2)

               IF (MaintainAuxArrays) THEN
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1) - mbc - 1
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2) + mbc + 1
               ELSE
                  mGlobal(1:nDim,1)=mGlobal(1:nDim,1) - mbc
                  mGlobal(1:nDim,2)=mGlobal(1:nDim,2) + mbc
               END IF
               Neighbors=Neighbors .OR. BoxOverlap(child%box%mGlobal, mGlobal)
            END DO
         END DO
      END DO

   END FUNCTION Neighbors

   !> @}


   !> @name Routines for manipulating node box indices
   !! @{

   !> Get indices of parent box in node's level
   !! @param node node object
   !! @param mB parent box's bounds
   !! @param level level of node's parent
   FUNCTION GetChildMGlobal(node,mB,level)
      TYPE(NodeDef), POINTER :: node
      INTEGER, DIMENSION(3,2) :: GetChildMGlobal, mB
      INTEGER :: level
      GetChildmGlobal=1
!      GetChildmGlobal(nDim+1:3,:)=1
      GetChildMGlobal(1:nDim,1)=(node%box%mGlobal(1:nDim,1)-2+mB(1:nDim,1))*levels(level)%CoarsenRatio+1
      GetChildMGlobal(1:nDim,2)=(node%box%mGlobal(1:nDim,1)-1+mB(1:nDim,2))*levels(level)%CoarsenRatio
   END FUNCTION GetChildMGlobal

   !> Transform indices of node child box into node's level
   !! @param node node object
   !! @param childmGlobal child box indices
   !! @param level node's level
   FUNCTION GetChildMBounds(node,childmGlobal,level)
      TYPE(NodeDef), POINTER :: node
      INTEGER, DIMENSION(3,2) :: GetChildMBounds, childmGlobal
      INTEGER :: level
      INTEGER, DIMENSION(3,2) :: work_array

!      GetChildmBounds(nDim+1:3,:)=1
      GetChildmBounds=1
!PRINT "('GetChildMBounds(', i1 ,')::ChildmGlobal = [', 6i4, '].')", MPI_id, ChildmGlobal
!PRINT "('GetChildMBounds(', i1 ,')::node%box%mGlobal = [', 6i4, '].')", MPI_id, node%box%mGlobal
!PRINT "('GetChildMBounds(', i1 ,')::CoarsenRatio = ', i4, '.')", MPI_id, levels(level)%CoarsenRatio
!      work_array = 1
!      work_array(1:nDim,1)=(childmGlobal(1:nDim,1)-1)/levels(level)%CoarsenRatio - node%box%mGlobal(1:nDim,1)+2
!      work_array(1:nDim,2)=(childmGlobal(1:nDim,2)  )/levels(level)%CoarsenRatio - node%box%mGlobal(1:nDim,1)+1
!PRINT "('GetChildMBounds(', i1 ,')::work_array = [', 6i4, '].')", MPI_id, work_array
!PRINT *

      GetChildMBounds(1:nDim,1)=(childmGlobal(1:nDim,1)-1)/levels(level)%CoarsenRatio - node%box%mGlobal(1:nDim,1)+2
      GetChildMBounds(1:nDim,2)=(childmGlobal(1:nDim,2)  )/levels(level)%CoarsenRatio - node%box%mGlobal(1:nDim,1)+1
   END FUNCTION GetChildMBounds
   !> @}

   !> @name Routines for manipulating node boxes
   !! @{

   !> Creates a node box
   !! @param box_array indices of new box
   !! @param node_box returns pointer to created box
   !! @param proc_id processor location of new box
   SUBROUTINE CreateNodeBox(box_array, node_box, proc_id)

      INTEGER, DIMENSION(3,2) :: box_array
      INTEGER, OPTIONAL :: proc_id
      TYPE(NodeBox), POINTER :: node_box

      INTEGER :: iErr


!      IF (ASSOCIATED(node_box)) THEN
!         DEALLOCATE(node_box)
!         NULLIFY(node_box)
!      END IF

      ALLOCATE(node_box, STAT=iErr)

      IF (iErr /= 0) THEN
         PRINT *, "AppendNodeBox() error:  unable to allocate new NodeBox structure."
         STOP
      END IF

      node_box%mGlobal = box_array
      IF (PRESENT(proc_id)) THEN
         node_box%MPI_id = proc_id
         !          PRINT "('CreateNodeBox([', 6i4, '], ', i2, ') done.')", box_array, proc_id
      ELSE
         !          PRINT "('CreateNodeBox([', 6i4, '], ', i2, ') done.')", box_array, MPI_id
      END IF

   END SUBROUTINE CreateNodeBox

   !> Destroys a node box
   !! @param node_box box to destroy
   SUBROUTINE DestroyNodeBox(node_box)

      TYPE(NodeBox), POINTER :: node_box

      IF (ASSOCIATED(node_box)) THEN
         DEALLOCATE(node_box)
         NULLIFY(node_box)
      END IF

   END SUBROUTINE DestroyNodeBox

   !> Routine to match two boxes
   !! @param box1 box 1
   !! @param box2 box 2
   FUNCTION MatchBox(box1,box2)
      TYPE(NodeBox) :: box1,box2
      LOGICAL :: MatchBox
      MatchBox=box1%MPI_ID==box2%MPI_ID .AND. ALL(box1%mGlobal==box2%mGlobal)
   END FUNCTION MatchBox

   !> @}

   !> @name Routines for manipulating nodeboxlists
   !! @{ 

   !> Adds a new NodeBoxList structure to the the block list passed in.
   !! @param last_box The NodeBox list to which the new block will be appended.
   !! @param first_box An optional MessageBlock pointer that will point to a freshly-created MessageBlock list.
   !! @param current_box Another optional MessageBlock pointer that will point to a freshly-created MessageBlock list.
   SUBROUTINE AddNodeBoxToList(last_box, first_box, current_box)

      TYPE(NodeBoxList), POINTER :: last_box
      TYPE(NodeBoxList), POINTER, OPTIONAL :: first_box
      TYPE(NodeBoxList), POINTER, OPTIONAL :: current_box

      INTEGER :: iErr


      iErr = 0

      IF (.NOT. ASSOCIATED(last_box)) THEN
         ALLOCATE(last_box, STAT=iErr)
         IF (PRESENT(first_box))  first_box => last_box
         IF (PRESENT(current_box))  current_box => first_box

      ELSE

         IF (ASSOCIATED(last_box%next)) THEN
            PRINT *, "AddNodeBox error: last_box%next associated."
            STOP
         END IF

         ALLOCATE(last_box%next, STAT=iErr)
         last_box => last_box%next

      END IF

      IF (iErr /= 0) THEN
         PRINT *, "AddNodeBox() error: unable to allocate new node box."
         STOP
      END IF

      NULLIFY(last_box%next)

   END SUBROUTINE AddNodeBoxToList

   !> Deletes elements from a NodeBox list structure, starting with the referenced element.
   !! @param box_list A reference to a node box list list.
   RECURSIVE SUBROUTINE ClearNodeBoxList(box_list)

      TYPE(NodeBoxList), POINTER :: box_list


      IF (.NOT. ASSOCIATED(box_list))  RETURN

      IF (ASSOCIATED(box_list%next))  CALL ClearNodeBoxList(box_list%next)

      DEALLOCATE(box_list)
      NULLIFY(box_list)

   END SUBROUTINE ClearNodeBoxList

   !! Counts the number of elements in a NodeBox list.
   !! @param box_list The list whose elements are to be counted.
   INTEGER FUNCTION NodeBoxCount(box_list)

      TYPE(NodeBoxList), POINTER :: box_list

      INTEGER :: work_count
      TYPE(NodeBoxList), POINTER :: iter


      iter => box_list
      work_count = 0

      DO WHILE (ASSOCIATED(iter))
         work_count = work_count + 1
         iter => iter%next
      END DO

      NodeBoxCount = work_count

   END FUNCTION NodeBoxCount

   !> Locates the parent node of the specified box, or returns null.
   !! @param level The level of the grid whose parent is being sought.
   !! @param mGlobal The dimensions of the grid whose parent is being sought.
   !! @param parent A pointer that will be associated with a parent node.
   SUBROUTINE FindParent(level, mGlobal, parent)
       INTEGER :: level
       INTEGER, DIMENSION(MAX_DIMS, 2) :: mGlobal
       TYPE(NodeDef), POINTER :: parent

       TYPE(NodeDefList), POINTER :: iter


       NULLIFY(parent)

       ! Search through local nodes for parent.
       iter => Nodes(level - 1)%p

       DO WHILE (ASSOCIATED(iter))

          IF (ALL(mGlobal(:,1) >= iter%self%box%mGlobal(:,1)) .AND. &
              ALL(mGlobal(:,2) <= iter%self%box%mGlobal(:,2))) THEN
              parent => iter%self
              RETURN
          END IF

       END DO

       ! search through external nodes for parent.
       iter => ExternalNodes(level - 1)%p

       DO WHILE (ASSOCIATED(iter))

          IF (ALL(mGlobal(:,1) >= iter%self%box%mGlobal(:,1)) .AND. &
              ALL(mGlobal(:,2) <= iter%self%box%mGlobal(:,2))) THEN
              parent => iter%self
              RETURN
          END IF

       END DO

   END SUBROUTINE FindParent

   !> Seeks the parent node for the specified grid, but kills the program if it can't be found.
   !! @param level The level of the grid whose parent is being sought.
   !! @param mGlobal The dimensions of the grid whose parent is being sought.
   !! @param parent A pointer that will be associated with a parent node.
   !! @param caller A string containing the calling process (used only if a parent node isn't found). 
   SUBROUTINE StrictFindParent(level, mGlobal, parent, caller)

       INTEGER :: level
       INTEGER, DIMENSION(MAX_DIMS, 2) :: mGlobal
       TYPE(NodeDef), POINTER :: parent
       CHARACTER(LEN=*) :: caller


       ! Search for parent node.
       CALL FindParent(level, mGlobal, parent)

       ! Kill program with detailed error info if node is not found.
       IF (.NOT. ASSOCIATED(parent)) THEN
           PRINT *, caller, "::StrictFindNode() error: parent not found."
           PRINT "('StrictFindParent(level = ', i3, ') failed on box [', 6I5, ']')", level, mGlobal
           STOP
       END IF
! Comment test
   END SUBROUTINE StrictFindParent

   !> @}

END MODULE TreeDeclarations

