!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    tree_level_comms.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @defgroup TreeComms Tree Communications
!! @ingroup TreeOps

!> @file tree_level_comms.f90
!! brief Main file for module TreeLevelComms

!> @defgroup TreeLevelComms Tree Level Communications
!> @brief Main module for scheduling AMR related tree communication
!! @ingroup TreeComms

!> Main module for scheduling AMR related tree communication
!! @ingroup TreeLevelComms
MODULE TreeLevelComms
  USE TreeDeclarations
  USE TreeParsing
  USE CommunicationDeclarations
  USE GlobalDeclarations
  USE DistributionDeclarations
  USE Timing
  IMPLICIT NONE


  !Interlevel Sends To Children
  PUBLIC PostSendGridsToChildren, PostSendOverlapsNeighbors, PostSendOverlapsToOldNodesChildren, PostSendOverlapsToNodesOldChildren

  !Interlevel Receives From Parents
  PUBLIC PostRecvGridsFromParents, PostRecvOverlapsNeighbors, PostRecvOldNodeOverlaps

  !Intralevel Sends/Recvs from neighbors/overlaps
  PUBLIC PostSendNeighboringChildren, PostSendOverlappingChildrenToOldNodes, PostSendOverlappingChildrenToNewNodes, &
       PostRecvNeighboringChildren, PostRecvOverlappingChildrenFromOldNodes, PostRecvOverlappingChildrenFromNewNodes

  !Interlevel Send Completions to Children
  PUBLIC CompSendGridsToChildren, CompSendOverlapsNeighbors, CompSendOverlapsToOldNodesChildren, CompSendOverlapsToNodesOldChildren

  !Interlevel Receive Completions from Parents
  PUBLIC CompRecvGridsFromParents, CompRecvOverlapsNeighbors, CompRecvOldNodeOverlaps

  !Intralevel Sends/Recvs Completions from neighbors/overlaps
  PUBLIC CompSendNeighboringChildren, CompSendOverlappingChildrenToOldNodes, CompSendOverlappingChildrenToNewNodes, &
       CompRecvNeighboringChildren, CompRecvOverlappingChildrenFromOldNodes, CompRecvOverlappingChildrenFromNewNodes 
  
  PUBLIC PackTest
  PRIVATE
CONTAINS


  !> @name InheritNeighborsChildrenComms Routines required for InheritNeighborsChildren
  !! @{

  SUBROUTINE PostSendNeighboringChildren(n) 
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    TYPE(NodeDefList), POINTER :: nodelist, neighborlist
    TYPE(NodeDef), POINTER :: node, neighbor
    CALL StartTimer(iiSendNeighboringChildren, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendNeighboringChildren,n)%p, TRANSMIT_NEIGHBORING_CHILDREN,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendNeighboringChildren,n)%p

    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       neighborlist=>node%neighbors 
       DO WHILE (associated(neighborlist))
          neighbor=>neighborlist%self 
          IF (neighbor%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup,neighbor%box%MPI_ID, message) !This ensures that every neighbor gets a message - even if there is nothing to send
             CALL SendNeighboringChildren(message,node,neighbor)
          END IF
          neighborlist=>neighborlist%next           
       END DO
       nodelist=>nodelist%next 
    END DO
    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)
    CALL StopTimer(iiSendNeighboringChildren, n)
  END SUBROUTINE PostSendNeighboringChildren

  SUBROUTINE CompSendNeighboringChildren(n) 
    INTEGER :: n
    CALL StartTimer(iiSendNeighboringChildren, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendNeighboringChildren,n)%p)
    CALL StopTimer(iiSendNeighboringChildren, n)
  END SUBROUTINE CompSendNeighboringChildren

  SUBROUTINE PostRecvNeighboringChildren(n) 
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    TYPE(NodeDefList), POINTER :: nodelist, neighborlist
    TYPE(NodeDef), POINTER :: node, neighbor
    CALL StartTimer(iiRecvNeighboringChildren, n)
    CALL CreateMessageGroup(StageMessageGroups(iRecvNeighboringChildren,n)%p, TRANSMIT_NEIGHBORING_CHILDREN,STAGE_RECV,n)
    MessageGroup=>StageMessageGroups(iRecvNeighboringChildren,n)%p

    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       neighborlist=>node%neighbors 
       DO WHILE (associated(neighborlist))
          neighbor=>neighborlist%self 
          IF (neighbor%box%MPI_ID /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup,neighbor%box%MPI_ID, message) !This ensures that every neighbor gets a message - even if there is nothing to send
          neighborlist=>neighborlist%next
       END DO
       nodelist=>nodelist%next 
    END DO
    CALL StopTimer(iiRecvNeighboringChildren, n)
  END SUBROUTINE PostRecvNeighboringChildren

  SUBROUTINE CompRecvNeighboringChildren(n) 
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group     
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvNeighboringChildren, n)

    sm_group => StageMessageGroups(iRecvNeighboringChildren,n)%p
    CALL MGBlockOnFirstMessages(sm_group, message)

    DO WHILE (ASSOCIATED(message))
       CALL RecvNeighboringChildren(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO

    CALL DestroyMessageGroup(StageMessageGroups(iRecvNeighboringChildren,n)%p)
    CALL StopTimer(iiRecvNeighboringChildren, n)
  END SUBROUTINE CompRecvNeighboringChildren

  !> @}

  !> @name InheritOldNodeOverlapsChildrenComms Routines required for InheritOldNodeOverlapsChildren 
  !! @{

  SUBROUTINE PostSendOverlappingChildrenToNewNodes(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiSendOverlappingChildrenToNewNodes, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendOverlappingChildrenToNewNodes,n)%p, TRANSMIT_OLC_TO_NEW_NODES,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendOverlappingChildrenToNewNodes,n)%p

!IF ((n == 1) .AND. (MPI_id == 7)) PRINT "('Proc ', i1, ' posted OLC_TNN send for level ', i2, '.')", MPI_id, n


    nodelist=>OldNodes(n)%p 

    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       overlaplist=>node%overlaps 

       DO WHILE (associated(overlaplist))
          overlap=>overlaplist%self 
!IF ((n == 1) .AND. (MPI_id == 7)) PRINT "('PostSendOverlappingChildren::overlap = [', i1, '][', 6i4, '].')", overlap%box%MPI_id, overlap%box%mGlobal
          IF (overlap%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, overlap%box%MPI_ID, message)
!IF ((n == 1) .AND. (MPI_id == 7)) PRINT "('Proc ', i1, ', level ', i2, ' added OLC_TNN send for remote proc ', i1, '.')", MPI_id, n, overlap%box%MPI_id
             CALL SendOverlappingChildren(message,node,overlap)
          END IF
          overlaplist=>overlaplist%next
       END DO
       nodelist=>nodelist%next
    END DO
    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)

    CALL StopTimer(iiSendOverlappingChildrenToNewNodes, n)
  END SUBROUTINE PostSendOverlappingChildrenToNewNodes


  SUBROUTINE CompSendOverlappingChildrenToNewNodes(n) 
    INTEGER :: n
    CALL StartTimer(iiSendOverlappingChildrenToNewNodes, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendOverlappingChildrenToNewNodes,n)%p)
    CALL StopTimer(iiSendOverlappingChildrenToNewNodes, n)
  END SUBROUTINE CompSendOverlappingChildrenToNewNodes

  SUBROUTINE PostRecvOverlappingChildrenFromOldNodes(n) 
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvOverlappingChildrenFromOldNodes, n)

    CALL CreateMessageGroup(StageMessageGroups(iRecvOverlappingChildrenFromOldNodes,n)%p, TRANSMIT_OLC_TO_NEW_NODES,STAGE_RECV,n)
    MessageGroup=>StageMessageGroups(iRecvOverlappingChildrenFromOldNodes,n)%p

    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       overlaplist=>node%overlaps 
       DO WHILE (associated(overlaplist))
          overlap=>overlaplist%self 
          IF (overlap%box%MPI_ID /= MPI_ID) THEN
              CALL ExtractMessageFromGroup(MessageGroup, overlap%box%MPI_ID, message)
          END IF
          overlaplist=>overlaplist%next 
       END DO
       nodelist=>nodelist%next
    END DO
    CALL StopTimer(iiRecvOverlappingChildrenFromOldNodes, n)
  END SUBROUTINE PostRecvOverlappingChildrenFromOldNodes

  SUBROUTINE CompRecvOverlappingChildrenFromOldNodes(n) 
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group 
    TYPE(PackedMessage), POINTER :: message

    CALL StartTimer(iiRecvOverlappingChildrenFromOldNodes, n)
    sm_group=>StageMessageGroups(iRecvOverlappingChildrenFromOldNodes,n)%p
    CALL MGBlockOnFirstMessages(sm_group, message)

    DO WHILE (ASSOCIATED(message))
       CALL RecvOverlappingChildrenFromOldNodes(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO

    CALL DestroyMessageGroup(StageMessageGroups(iRecvOverlappingChildrenFromOldNodes,n)%p)
    CALL StopTimer(iiRecvOverlappingChildrenFromOldNodes, n)

  END SUBROUTINE CompRecvOverlappingChildrenFromOldNodes


  !> @}

  !> @name InheritNewnodeOverlapsChildrenComms Routines required for InheritNewNodeOverlapsChildren 
  !! @{


  !> Sends overlapping child nodes for all n-th level nodes
  !! @param n    nth level nodes  
  SUBROUTINE PostSendOverlappingChildrenToOldNodes(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiSendOverlappingChildrenToOldNodes, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendOverlappingChildrenToOldNodes,n)%p, TRANSMIT_OLC_TO_OLD_NODES,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendOverlappingChildrenToOldNodes,n)%p

    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       overlaplist=>node%overlaps 
       DO WHILE (associated(overlaplist))
          overlap=>overlaplist%self 
          IF (overlap%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, overlap%box%MPI_ID, message)
             CALL SendOverlappingChildren(message,node,overlap)
          END IF
          overlaplist=>overlaplist%next
       END DO
       nodelist=>nodelist%next
    END DO
    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)
    CALL StopTimer(iiSendOverlappingChildrenToOldNodes, n)

  END SUBROUTINE PostSendOverlappingChildrenToOldNodes

  SUBROUTINE CompSendOverlappingChildrenToOldNodes(n) 
    INTEGER :: n
    CALL StartTimer(iiSendOverlappingChildrenToOldNodes, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendOverlappingChildrenToOldNodes,n)%p)
    CALL StopTimer(iiSendOverlappingChildrenToOldNodes, n)
  END SUBROUTINE CompSendOverlappingChildrenToOldNodes


  SUBROUTINE PostRecvOverlappingChildrenFromNewNodes(n) 
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvOverlappingChildrenFromNewNodes, n)
    CALL CreateMessageGroup(StageMessageGroups(iRecvOverlappingChildrenFromNewNodes,n)%p, TRANSMIT_OLC_TO_OLD_NODES,STAGE_RECV,n)
    MessageGroup=>StageMessageGroups(iRecvOverlappingChildrenFromNewNodes,n)%p

    nodelist=>OldNodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       overlaplist=>node%overlaps 
       DO WHILE (associated(overlaplist))
          overlap=>overlaplist%self 

!IF ((n == 1) .AND. (MPI_id == 0)) PRINT "('PostRecvOverlappingChildren::overlap = [', i1, '][', 6i4, '].')", overlap%box%MPI_id, overlap%box%mGlobal
          IF (overlap%box%MPI_ID /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup, overlap%box%MPI_ID, message)
          overlaplist=>overlaplist%next 
       END DO
       nodelist=>nodelist%next
    END DO
    CALL StopTimer(iiRecvOverlappingChildrenFromNewNodes, n)
  END SUBROUTINE PostRecvOverlappingChildrenFromNewNodes

  SUBROUTINE CompRecvOverlappingChildrenFromNewNodes(n) 
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group 
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvOverlappingChildrenFromNewNodes, n)
    sm_group=>StageMessageGroups(iRecvOverlappingChildrenFromNewNodes,n)%p
    CALL MGBlockOnFirstMessages(sm_group, message)

    DO WHILE (ASSOCIATED(message))
       CALL RecvOverlappingChildrenFromNewNodes(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO

    CALL DestroyMessageGroup(StageMessageGroups(iRecvOverlappingChildrenFromNewNodes,n)%p)

    CALL StopTimer(iiRecvOverlappingChildrenFromNewNodes, n)
  END SUBROUTINE CompRecvOverlappingChildrenFromNewNodes

  !> @}

  !> @name ChildGridComms Routines required to communicate new nodes to child processors
  !! @{ 


  !> @brief Send child nodes of all n-th level nodes the NodeBox grids.
  !! @param n - level
  SUBROUTINE PostSendGridsToChildren(n)
    INTEGER :: n, i
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    LOGICAL, DIMENSION(:,:), POINTER  :: ChildChildArray
    LOGICAL, DIMENSION(:), POINTER :: GrandChildArray
    TYPE(NodeDefList), POINTER :: nodelist,childlist
    TYPE(NodeDef), POINTER :: node,child
    CALL StartTimer(iiSendGridsToChildren, n)

    levels(n+1)%MPI_COMM=levels(n)%MPI_COMM
    CALL CreateMessageGroup(StageMessageGroups(iSendGridsToChildren,n)%p, TRANSMIT_CHILD_GRIDS, STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendGridsToChildren,n)%p

    IF (ASSOCIATED(ChildProcs(n)%p)) THEN
       DO i=1,size(ChildProcs(n)%p)          
          IF (ChildPRocs(n)%p(i) /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup, ChildProcs(n)%p(i), Message)
       END DO
    END IF
    nodelist=>Nodes(n)%p 

    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       childlist=>node%children 

       DO WHILE (associated(childlist))
          child=>childlist%self 
          IF (child%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, child%box%MPI_ID, Message)
             CALL SendGridToChild(message,child)
          END IF
          childlist=>childlist%next
       END DO

       nodelist=>nodelist%next
    END DO

    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)


    CALL StopTimer(iiSendGridsToChildren, n)    

  END SUBROUTINE PostSendGridsToChildren

  SUBROUTINE CompSendGridsToChildren(n) 

    INTEGER :: n
    CALL StartTimer(iiSendGridsToChildren, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendGridsToChildren,n)%p)
    CALL StopTimer(iiSendGridsToChildren, n)                      
  END SUBROUTINE CompSendGridsToChildren


  SUBROUTINE PostRecvGridsFromParents(n)
    INTEGER :: n, i
    TYPE(StageMessageGroup), POINTER ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvGridsFromParents, n)
    ! Creates a message group to manage this processor's open receives.
    CALL CreateMessageGroup(StageMessageGroups(iRecvGridsFromParents,n)%p, TRANSMIT_CHILD_GRIDS, STAGE_RECV,n-1)
    MessageGroup=>StageMessageGroups(iRecvGridsFromParents,n)%p
    
    IF (ASSOCIATED(ParentProcs(n)%p)) THEN
       DO i=1,size(ParentProcs(n)%p)
          IF (ParentProcs(n)%p(i) /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup,ParentProcs(n)%p(i),message)
       END DO
    END IF
    
    CALL StopTimer(iiRecvGridsFromParents, n)

  END SUBROUTINE PostRecvGridsFromParents

  SUBROUTINE CompRecvGridsFromParents(n)
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group
    TYPE(PackedMessage), POINTER :: message
    LOGICAL, DIMENSION(:), POINTER :: NewParentsArray
    CALL StartTimer(iiRecvGridsFromParents, n)

    sm_group => StageMessageGroups(iRecvGridsFromParents,n)%p

    ! Retrieve first message from group.
    CALL MGBlockOnFirstMessages(sm_group, message)

    ! Continue processing messages until there are no more open messages.
    DO WHILE (ASSOCIATED(message))
       CALL RecvGridsFromParent(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO


    ! Close all open handles and destroy the message group.
    CALL DestroyMessageGroup(StageMessageGroups(iRecvGridsFromParents,n)%p)
    CALL StopTimer(iiRecvGridsFromParents, n)
        

  END SUBROUTINE CompRecvGridsFromParents

  !> @}

  !> @name NewChildrenOverlapsNeighborsComms Routines required to communicate overlaps and neighbors to new children nodes.
  !! @{

  !> @brief Sends overlaps and neighbors to children of n-th level grids
  !! @param n - level
  SUBROUTINE PostSendOverlapsNeighbors(n) 
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, child
    TYPE(NodeDefList), POINTER :: childlist, nodelist
    TYPE(StageMessageGroup), POINTER ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiSendOverlapsNeighbors, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendOverlapsNeighbors,n)%p, TRANSMIT_OVERLAPS_NEIGHBORS,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendOverlapsNeighbors,n)%p      
    nodelist=>Nodes(n)%p 

    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       childlist=>node%children 

       DO WHILE (associated(childlist))
          child=>childlist%self 
          IF (child%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, child%box%MPI_ID, Message)
             CALL SendOverlapsNeighborsToChild(message,child) 
          END IF
          childlist=>childlist%next
       END DO

       nodelist=>nodelist%next 
    END DO

    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)

    CALL StopTimer(iiSendOverlapsNeighbors, n)           

  END SUBROUTINE PostSendOverlapsNeighbors

  SUBROUTINE CompSendOverlapsNeighbors(n) 
    INTEGER :: n
    CALL StartTimer(iiSendOverlapsNeighbors, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendOverlapsNeighbors,n)%p)
    CALL StopTimer(iiSendOverlapsNeighbors, n)
  END SUBROUTINE CompSendOverlapsNeighbors

  SUBROUTINE PostRecvOverlapsNeighbors(n)
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: Message
    TYPE(NodeDefList), POINTER :: nodelist
    TYPE(NodeDef), POINTER :: node
    CALL StartTimer(iiRecvOverlapsNeighbors, n)

    CALL CreateMessageGroup(StageMessageGroups(iRecvOverlapsNeighbors,n)%p, TRANSMIT_OVERLAPS_NEIGHBORS,STAGE_RECV,n-1)
    MessageGroup=>StageMessageGroups(iRecvOverlapsNeighbors,n)%p

    nodelist=>Nodes(n)%p
    DO WHILE (ASSOCIATED(nodelist))
       node=>nodelist%self
       IF (node%parent%box%MPI_ID /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup,node%parent%box%MPI_ID, message)
       nodelist=>nodelist%next
    END DO

    CALL StopTimer(iiRecvOverlapsNeighbors, n)         
  END SUBROUTINE PostRecvOverlapsNeighbors

  SUBROUTINE CompRecvOverlapsNeighbors(n)
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group
    TYPE(PackedMessage), POINTER :: message
    TYPE(NodeDefList), POINTER :: testlist, testneighborlist
    TYPE(NodeDef), POINTER :: testnode
    CALL StartTimer(iiRecvOverlapsNeighbors,n)
    sm_group=>StageMessageGroups(iRecvOverlapsNeighbors,n)%p

    CALL MGBlockOnFirstMessages(sm_group, message)

    DO WHILE (ASSOCIATED(message))
       CALL RecvOverlapsNeighbors(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO

    CALL DestroyMessageGroup(StageMessageGroups(iRecvOverlapsNeighbors,n)%p)
    CALL StopTimer(iiRecvOverlapsNeighbors,n)      
  END SUBROUTINE CompRecvOverlapsNeighbors
  !> @}


  !> @name OldChildGridComms Routines required to communicate new overlaps to old child grids.
  !! @brief There are two basic sending flavors depending on whether old children are children of old nodes 
  !! or old children of the current nodes.
  !! The receives however are the same - since the children are old level n+1 grids.
  !! @{

  SUBROUTINE PostSendOverlapsToOldNodesChildren(n) 

    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, child
    TYPE(NodeDefList), POINTER :: childlist, nodelist
    TYPE(StageMessageGroup), POINTER :: MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiSendOverlapsToOldNodesChildren, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendOverlapsToOldNodesChildren,n)%p, TRANSMIT_OLD_NODE_OVERLAPS,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendOverlapsToOldNodesChildren,n)%p

    nodelist=>OldNodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       childlist=>node%children 

       DO WHILE (associated(childlist))
          child=>childlist%self 
          IF (child%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, child%box%MPI_ID, Message)
             CALL SendOverlapsToChild(message, child) 
          END IF
          childlist=>childlist%next
       END DO
       nodelist=>nodelist%next 
    END DO

    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)

    CALL StopTimer(iiSendOverlapsToOldNodesChildren, n)

  END SUBROUTINE PostSendOverlapsToOldNodesChildren


  SUBROUTINE PostSendOverlapsToNodesOldChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, child
    TYPE(NodeDefList), POINTER :: childlist, nodelist
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiSendOverlapsToNodesOldChildren, n)

    CALL CreateMessageGroup(StageMessageGroups(iSendOverlapsToNodesOldChildren,n)%p, TRANSMIT_OLD_NODE_OVERLAPS,STAGE_SEND,n)
    MessageGroup=>StageMessageGroups(iSendOverlapsToNodesOldChildren,n)%p

    nodelist=>Nodes(n)%p 

    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       childlist=>node%oldchildren 

       DO WHILE (associated(childlist))
          child=>childlist%self 
          IF (child%box%MPI_ID /= MPI_ID) THEN
             CALL ExtractMessageFromGroup(MessageGroup, child%box%MPI_ID, Message)
             CALL SendOverlapsToChild(message, child) 
          END IF
          childlist=>childlist%next
       END DO
       nodelist=>nodelist%next 
    END DO

    CALL SendTerminationBox(MessageGroup)
    CALL CloseMessageGroup(MessageGroup)

    CALL StopTimer(iiSendOverlapsToNodesOldChildren, n)
  END SUBROUTINE PostSendOverlapsToNodesOldChildren

  SUBROUTINE CompSendOverlapsToOldNodesChildren(n)
    INTEGER :: n
    CALL StartTimer(iiSendOverlapsToOldNodesChildren, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendOverlapsToOldNodesChildren,n)%p)
    CALL StopTimer(iiSendOverlapsToOldNodesChildren, n)
  END SUBROUTINE CompSendOverlapsToOldNodesChildren


  SUBROUTINE CompSendOverlapsToNodesOldChildren(n)
    INTEGER :: n
    CALL StartTimer(iiSendOverlapsToNodesOldChildren, n)
    CALL DestroyMessageGroup(StageMessageGroups(iSendOverlapsToNodesOldChildren,n)%p)
    CALL StopTimer(iiSendOverlapsToNodesOldChildren, n)

  END SUBROUTINE CompSendOverlapsToNodesOldChildren

  SUBROUTINE PostRecvOldNodeOverlaps(n)
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer ::MessageGroup
    TYPE(PackedMessage), POINTER :: message
    TYPE(NodeDefList), POINTER :: nodelist
    TYPE(NodeDef), POINTER :: node

    CALL StartTimer(iiRecvOldNodeOverlaps,n)

    CALL CreateMessageGroup(StageMessageGroups(iRecvOldNodeOverlaps,n)%p, TRANSMIT_OLD_NODE_OVERLAPS,STAGE_RECV,n-1)
    MessageGroup=>StageMessageGroups(iRecvOldNodeOverlaps,n)%p
    nodelist=>OldNodes(n)%p

    DO WHILE (ASSOCIATED(nodelist))
       node=>nodelist%self
       IF (node%parent%box%MPI_ID /= MPI_ID) CALL ExtractMessageFromGroup(MessageGroup,node%parent%box%MPI_ID, message)
       nodelist=>nodelist%next
    END DO
    CALL StopTimer(iiRecvOldNodeOverlaps,n)
  END SUBROUTINE PostRecvOldNodeOverlaps

  SUBROUTINE CompRecvOldNodeOverlaps(n)
    INTEGER :: n
    TYPE(StageMessageGroup), Pointer :: sm_group
    TYPE(PackedMessage), POINTER :: message
    CALL StartTimer(iiRecvOldNodeOverlaps,n)

    sm_group=>StageMessageGroups(iRecvOldNodeOverlaps,n)%p
    CALL MGBlockOnFirstMessages(sm_group, message)

    DO WHILE (ASSOCIATED(message))
       CALL RecvOldNodeOverlaps(message)
       CALL MGBlockOnFirstMessages(sm_group, message)
    END DO

    CALL DestroyMessageGroup(StageMessageGroups(iRecvOldNodeOverlaps,n)%p)

    CALL StopTimer(iiRecvOldNodeOverlaps,n)        
  END SUBROUTINE CompRecvOldNodeOverlaps
  !> @}


   SUBROUTINE PackTest

      TYPE(StageMessageGroup), POINTER :: sm_group
      TYPE(PackedMessage), POINTER :: message
!      REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: payload
!      REAL, POINTER, DIMENSION(:,:,:,:) :: payload
      INTEGER, POINTER, DIMENSION(:,:,:,:) :: payload
      INTEGER :: proc
      INTEGER :: i,j,k,m,counter


      ALLOCATE(payload(8,8,8,8))
!      payload = -1.d0
!      payload = -1.0
      payload = -1

      IF (MODULO(MPI_id, 2) == 0) THEN

         NULLIFY(message)

         CALL CreateMessageGroup(sm_group, 5000, STAGE_SEND,0)


         NULLIFY(message)

         DO proc = 1, MPI_np - 1, 2
            CALL ExtractMessageFromGroup(sm_group, proc, message)
         END DO

         DO proc = 1, MPI_np - 1, 2
            CALL ExtractMessageFromGroup(sm_group, proc, message)
!            payload = REAL(MPI_id)
!            payload = REAL(MPI_id)
!            payload = MPI_id
            counter = 0
            DO m = 1, 8
            DO k = 1, 8
            DO j = 1, 8
            DO i = 1, 8
                counter = counter + 1
!                payload(i,j,k,m) = REAL(counter, KIND=qPrec)
                payload(i,j,k,m) = counter
            END DO
            END DO
            END DO
            END DO
!            CALL PrintPayload(payload)
            CALL PackData(message, payload)
         END DO

         CALL CloseMessageGroup(sm_group)
         CALL DestroyMessageGroup(sm_group)

      ELSE

         CALL CreateMessageGroup(sm_group, 5000, STAGE_RECV,0)

         NULLIFY(message)

         DO proc = 0, MPI_np - 2, 2
            CALL ExtractMessageFromGroup(sm_group, proc, message)
!PRINT *, "PackTest::message%closed = ", message%closed
         END DO

         NULLIFY(message)

         CALL MGBlockOnFirstMessages(sm_group, message)

!PRINT *, "PackTest::done message%closed = ", message%closed
         DO WHILE (ASSOCIATED(message))

!PRINT *, "remote_proc ", message%remote_proc, " nMessages = ", message%nMessages
!PRINT *, "remote_proc ", message%remote_proc, " last_block_size = ", message%last_block%buffer_size

!             payload = -1.d0
!             payload = -1.0
             payload = -1
             CALL UnpackData(message, payload)
             PRINT *, "message from ", message%remote_proc
             CALL PrintPayload(payload)
             CALL MGBlockOnFirstMessages(sm_group, message)
         END DO

         CALL DestroyMessageGroup(sm_group)

      END IF

      DEALLOCATE(payload)
      NULLIFY(payload)

   END SUBROUTINE PackTest

   SUBROUTINE PrintPayload(payload)
!       REAL(KIND=qPrec), DIMENSION(:,:,:,:) :: payload
!       REAL, DIMENSION(:,:,:,:) :: payload
       INTEGER, DIMENSION(:,:,:,:) :: payload
       INTEGER :: i,j,k,m
  
!       PRINT *, "proc ", MPI_id
!       PRINT *, "Average payload = ", SUM(payload) * 1.0 / SIZE(payload)
!       PRINT *, "Minimum payload = ", MINVAL(payload)
!       PRINT *, "Maximum payload = ", MAXVAL(payload)
!       PRINT *, "COUNT(0) = ", COUNT(payload == 0)
!       PRINT *, "COUNT(2) = ", COUNT(payload == 2)
!       PRINT *, "COUNT = ", SIZE(payload)
!       PRINT *
!       PRINT *, "proc ", MPI_id
!       PRINT *
!         DO l=1,4
!         DO k=1,4
!            PRINT *, "k=", k, "l=", l
!            DO j=1,4
!                PRINT "('  [', 4f8.5, ']')", payload(:,j,k,l)
!            END DO
!            PRINT *
!         END DO
!         END DO
       DO m=LBOUND(payload,4),UBOUND(payload,4)
       DO k=LBOUND(payload,3),UBOUND(payload,4)
       DO j=LBOUND(payload,2),UBOUND(payload,2)
!       DO i=LBOUND(payload,1),UBOUND(payload,1)
!           PRINT "('Proc ', i1, ' payload[', i2, ', ', i2, ', 1, ', i1, '] = ', f, '.')", MPI_id, i,j,m,payload(i,j,1,m)
            PRINT "('Proc ', i1, ' payload = [', 8I10, '].')", MPI_id, payload(:,j,k,m)
       END DO
       END DO
       END DO
   END SUBROUTINE PrintPayload

END MODULE TreeLevelComms

