!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    tree_level_ops.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir tree
!! @brief Contains amr-related modules for managing info structures.

!> @file tree_level_ops.f90
!! @brief Main file for module TreeLevelOps

!> @defgroup TreeOps Tree Operations
!! @ingroup AMR

!> @defgroup TreeLevelOps Tree Level Operations
!> @brief Main module for managing AMR related tree operations by level
!! @ingroup TreeOps

!> Main module for managing AMR related tree operations by level
!! @ingroup TreeLevelOps


MODULE TreeLevelOps
  USE TreeNodeOps
  USE Timing
  IMPLICIT NONE
  PRIVATE


  !Pair-wise tree operations   
  PUBLIC :: InheritNeighborsChildren, InheritOldNodeOverlapsChildren, InheritNewNodeOverlapsChildren, &
       InheritOverlapsOldChildren, InheritOverlapsNewChildren

  !Single node tree operations
  PUBLIC :: CreateChildrens, UpdateOverlaps, NullifyNeighbors, &
       AgeNodesChildren, AgeNodes, DestroyNodes, DestroyOldNodes

CONTAINS 


  !> @name Pair-wise Tree Operations
  !! @{

  !> @brief Looks to see what children of neighbors should be neighbors of children
  !! @param n level
  SUBROUTINE InheritNeighborsChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, neighbor
    TYPE(NodeDefList), POINTER :: nodelist, neighborlist
    CALL StartTimer(iInheritNeighborsChildren, n)
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self       
       neighborlist=>node%neighbors 
       DO WHILE (associated(neighborlist))
          neighbor=>neighborlist%self
          IF (neighbor%box%MPI_ID == MPI_ID) CALL InheritNeighborChildren(node,neighbor,n) !Have child grids locally
          neighborlist=>neighborlist%next
       END DO
       CALL InheritSelfNeighborChildren(node,n)
       nodelist=>nodelist%next 
    END DO
    CALL StopTimer(iInheritNeighborsChildren, n)
  END SUBROUTINE InheritNeighborsChildren


  !Overlaps are funny because they alternate between being old grids - and being neighbor grids
  !Inheritance of overlaps gets even trickier because on the second step when neighbors become overlaps
  !a node already has the overlaps' children.

  !(First step)
  !AddOverlap(Nodes%children,Nodes%Overlaps%Children) !InheritOverlapsChildren
  !AddOverlap(OldNodes%children, OldNodes%Overlaps%Children) !InheritOverlapsChildren

  !(Second Step)
  !AddOverlap(Nodes%children,Nodes%Overlaps%OldChildren) !InheritOverlapsOldChildren *No communication required!
  !AddOverlap(Nodes%OldChildren,Nodes%Overlaps%Children) !InheritOverlapsNewChildren


  !> @brief Looks to see what children of overlaps should be overlaps of children for both current and old nodes
  !! @param n level
  SUBROUTINE InheritOldNodeOverlapsChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    CALL StartTimer(iInheritOldNodeOverlapsChildren, n)
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))

       node=>nodelist%self 
       overlaplist=>node%overlaps 

       DO WHILE (associated(overlaplist))
          overlap=>overlaplist%self 
          IF (overlap%box%MPI_ID == MPI_ID) CALL InheritOverlapChildren(node,overlap,n)
          overlaplist=>overlaplist%next
       END DO

       nodelist=>nodelist%next 

    END DO
    CALL StopTimer(iInheritOldNodeOverlapsChildren, n)
  END SUBROUTINE InheritOldNodeOverlapsChildren

  SUBROUTINE InheritNewNodeOverlapsChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist

    CALL StartTimer(iInheritNewNodeOverlapsChildren, n)

    nodelist=>OldNodes(n)%p 

    DO WHILE (associated(nodelist))

       node=>nodelist%self 
       overlaplist=>node%overlaps 

       DO WHILE (associated(overlaplist))

          overlap=>overlaplist%self 
          IF (overlap%box%MPI_ID == MPI_ID) CALL InheritOverlapChildren(node,overlap,n)
          overlaplist=>overlaplist%next

       END DO

       nodelist=>nodelist%next 

    END DO

    CALL StopTimer(iInheritNewNodeOverlapsChildren, n)
  END SUBROUTINE InheritNewNodeOverlapsChildren

  !> @brief Looks to see what old children of overlaps should be overlaps of new children for current nodes
  !! @param n level
  SUBROUTINE InheritOverlapsOldChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist, externallist
    CALL StartTimer(iInheritOverlapsOldChildren, n)
    nodelist=>Nodes(n)%p
    DO WHILE (associated(nodelist))

       node=>nodelist%self
       overlaplist=>node%overlaps 

       DO WHILE (associated(overlaplist))

          overlap=>overlaplist%self 

!!! If we are here it is because we are on the second of two steps 
!!! in which case overlaps are our previous neighbors and
!!! and overlaps@oldchildren are the previous neighbors%children which we
!!! already were informed about. (Assuming we age the children of external nodes)
          CALL InheritOverlapOldChildren(node,overlap,n)
          overlaplist=>overlaplist%next
       END DO

       CALL InheritSelfOverlapOldChildren(node,n)
       nodelist=>nodelist%next 

    END DO

    CALL StopTimer(iInheritOverlapsOldChildren, n)

  END SUBROUTINE InheritOverlapsOldChildren

  !> @brief Looks to see what children of overlaps should be overlaps of oldchildren 
  !! @param n level
  SUBROUTINE InheritOverlapsNewChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist
    CALL StartTimer(iInheritOverlapsNewChildren, n)

    nodelist=>Nodes(n)%p

    DO WHILE (associated(nodelist))

       node=>nodelist%self
       overlaplist=>node%overlaps 

       DO WHILE (associated(overlaplist))

          overlap=>overlaplist%self 

!!! If we are here it is because we are on the second of two steps 
!!! in which case overlaps are our previous neighbors and
!!! and overlaps@oldchildren are the previous neighbors%children which we
!!! already were informed about. (Assuming we age the children of external nodes)
!!! We should only call this after receiving all of our neighbors children
          CALL InheritOverlapNewChildren(node,overlap,n)            

          overlaplist=>overlaplist%next
       END DO

       CALL InheritSelfOverlapNewChildren(node,n)
       nodelist=>nodelist%next 

    END DO

    CALL StopTimer(iInheritOverlapsNewChildren, n)
  END SUBROUTINE InheritOverlapsNewChildren
  !> @}


  !> @name Single-Node Tree Operations
  !! @{

  !> @brief Creates and distributes new children of node
  !! @param  n level
  SUBROUTINE CreateChildrens(n)
    INTEGER :: n
    TYPE(NodeDefList), POINTER :: nodelist
    TYPE(NodeDef), POINTER :: node
    CALL StartTimer(iCreateChildrens, n)
    nodelist=>Nodes(n)%p      
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL CreateChildren(node,n)
       nodelist=>nodelist%next
    END DO
    CALL StopTimer(iCreateChildrens, n)
  END SUBROUTINE CreateChildrens


  !> @brief Updates overlap list to point to neighbor list for setting ghost cells etc...
  !! @param  n level
  SUBROUTINE UpdateOverlaps(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    CALL StartTimer(iUpdateOverlaps, n)

    nodelist=>Nodes(n)%p 

    DO WHILE (associated(nodelist))             
       node=>nodelist%self 
       IF (.NOT. ASSOCIATED(node%overlaps, target=node%neighbors)) THEN
          CALL ClearNodeList(node%overlaps) 
          NULLIFY(node%lastoverlap)
          node%overlaps=>node%neighbors
          node%lastoverlap=>node%lastneighbor
       END IF
       nodelist=>nodelist%next
    END DO

    CALL StopTimer(iUpdateOverlaps, n)
  END SUBROUTINE UpdateOverlaps

  !> Nullifies neighbors (but doesn't kill the nodelist since the overlaps still point to the neighbors)
  !! @param n level
  SUBROUTINE NullifyNeighbors(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    CALL StartTimer(iNullifyNeighbors, n)
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       NULLIFY(node%neighbors)
       NULLIFY(node%lastneighbor)
       nodelist=>nodelist%next
    END DO
    CALL StopTimer(iNullifyNeighbors, n)

  END SUBROUTINE NullifyNeighbors


  !> Sets nodes oldchildpen to point to current children and clears currentchildren (and associated fields)
  !! @param n level    
  SUBROUTINE AgeNodesChildren(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    CALL StartTimer(iAgeNodesChildren, n)
    nodelist=>Nodes(n)%p
    DO WHILE (associated(nodelist))
       IF (.NOT. ASSOCIATED(nodelist%self)) THEN
          PRINT*, 'Error : no node associated in AgeNodesChildren'
          STOP
       END IF
       node=>nodelist%self 
       CALL AgeNodeChildren(node) 
       IF (n > -1 .AND. n < MaxLevel) CALL DeAllocChildFixups(node%info)
       nodelist=>nodelist%next
    END DO

!!!  We want to age children of external neighbor nodes so we can use them for overlaps!!!
    nodelist=>ExternalNodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       CALL AgeNodeChildren(node) 
       nodelist=>nodelist%next
    END DO

    CALL StopTimer(iAgeNodesChildren, n)
  END SUBROUTINE AgeNodesChildren

  !> Ages nodes on level n
  !! @param n level
  SUBROUTINE AgeNodes(n)
    INTEGER :: n
    TYPE(NodeDefList), POINTER :: nodelist
    TYPE(NodeDef), POINTER :: node
    CALL StartTimer(iBackUpNodes, n)

    CALL DestroyOldNodes(n)
    OldNodes(n)%p=>Nodes(n)%p
    OldExternalNodes(n)%p=>ExternalNodes(n)%p
    LastOldLocalNode(n)%p=>LastLocalNode(n)%p
    LastOldExternalNode(n)%p=>LastExternalNode(n)%p

    NULLIFY(Nodes(n)%p) 
    NULLIFY(ExternalNodes(n)%p)
    NULLIFY(LastLocalNode(n)%p)
    NULLIFY(LastExternalNode(n)%p)

    nodelist=>OldNodes(n)%p

    DO WHILE (ASSOCIATED(nodelist))
       node=>nodelist%self

       IF (.NOT. ASSOCIATED(node)) THEN
          PRINT* ,"clearing an unassociated node!"
          STOP
       END IF
       IF (.NOT. ASSOCIATED(node%overlaps, target = node%neighbors)) THEN
          CALL ClearNodeList(node%neighbors)
       ELSE
          NULLIFY(node%neighbors)
       END IF
       CALL ClearNodeList(node%overlaps)
       NULLIFY(node%lastoverlap, node%lastneighbor) !,node%lastneighbor, node%neighbors) !done in nullify neighbors
       nodelist=>nodelist%next
    END DO
    !Because our external children have overlaps and neighbors stored locally
    nodelist=>OldExternalNodes(n)%p
    DO WHILE (ASSOCIATED(nodelist))
       node=>nodelist%self
       IF (.NOT. ASSOCIATED(node)) THEN
          PRINT* ,"clearing an unassociated node!"
          STOP
       END IF
       !Don't need neighbors (and/or overlaps anymore
       IF (.NOT. ASSOCIATED(node%overlaps, target = node%neighbors)) THEN
          CALL ClearNodeList(node%neighbors)
       ELSE
          NULLIFY(node%neighbors)
       END IF
       CALL ClearNodeList(node%overlaps)
       NULLIFY(node%lastoverlap, node%lastneighbor) !,node%lastneighbor,node%neighbors)
       nodelist=>nodelist%next
    END DO

    CALL StopTimer(iBackUpNodes, n  )
 END SUBROUTINE AgeNodes

  !> Destroys all old nodes on level n
  !! @param n level    
  SUBROUTINE DestroyOldNodes(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist, temp
    CALL DestroyNodeList(OldNodes(n)%p)
    CALL DestroyNodeList(OldExternalNodes(n)%p)
    NULLIFY(LastOldLocalNode(n)%p, LastOldExternalNode(n)%p)
  END SUBROUTINE DestroyOldNodes

  !> Destroys all current nodes on level n
  !! @param n level    
  SUBROUTINE DestroyNodes(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist, temp
    CALL DestroyNodeList(Nodes(n)%p)
    CALL DestroyNodeList(ExternalNodes(n)%p)
    CALL DestroyNodeList(BackupNodes(n)%p)
    CALL DestroyNodeList(BackupExternalNodes(n)%p)
    NULLIFY(LastLocalNode(n)%p, LastExternalNode(n)%p)
  END SUBROUTINE DestroyNodes


  !> @}
END MODULE TreeLevelOps

