!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    tree_node_ops.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file tree_node_ops.f90
!! @brief Main file for module TreeNodeOps

!> @defgroup TreeNodeOps Tree Node Operations
!! @brief Module for performing AMR related node operations
!! @ingroup TreeOps

!> Module for performing AMR related node operations
!! @ingroup TreeNodeOps

MODULE TreeNodeOps
  USE GlobalDeclarations, ONLY : levels, nDim, lStressTest, nDomains, Domains, MPI_NP, MPI_ID, MaxLevel, BaseLevel
  USE DataDeclarations, ONLY : AllocChildFixups, DeAllocChildFixups, levelup
  USE DataInfoOps, ONLY : NewSubGrids
  USE TreeDeclarations
  USE DistributionControl
  IMPLICIT NONE
  !Pair-wise tree operations
  PUBLIC :: InheritNeighborChildren, InheritOverlapChildren, InheritOverlapOldChildren, InheritOverlapNewChildren

  !Self-wise tree operations
  PUBLIC :: InheritSelfNeighborChildren, InheritSelfOverlapOldChildren,InheritSelfOverlapNewChildren

  !Single node tree operations
  PUBLIC :: CreateChildren, AgeNodeChildren, BackupParent, BackupChildren, RestoreParent, RestoreChildren

CONTAINS


  !> @name Pair-wise tree operations
  !! @{

  !> @brief Looks to see what children of neighbor should be neighbors of children
  !! @param node node whose children need neighbors
  !! @param neighbor neighboring node 
  !! @param n level
  SUBROUTINE InheritNeighborChildren(node,neighbor,n)
    TYPE(NodeDef), POINTER :: node, child, neighbor, neighborchild
    TYPE(NodeDefList), POINTER :: childlist, neighborchildlist
    INTEGER :: n

    childlist=>node%children          

    DO WHILE (associated(childlist))

       child=>childlist%self 

       IF (NephewCanBeNeighbor(neighbor,child,n)) THEN
          neighborchildlist=>neighbor%children 

          DO WHILE (associated(neighborchildlist))
             neighborchild=>neighborchildlist%self 
             IF (Neighbors(child,neighborchild,n+1)) CALL AddNeighbor(child,neighborchild) 
             neighborchildlist=>neighborchildlist%next
          END DO

       END IF

       childlist=>childlist%next

    END DO

  END SUBROUTINE InheritNeighborChildren

  !> @brief Looks to see what children of overlap should be overlaps of children
  !! @param node node whose children need overlaps
  !! @param overlap overlapping node 
  !! @param n level
  SUBROUTINE InheritOverlapChildren(node,overlap,n)
    TYPE(NodeDef), POINTER :: node, child, overlap, overlapchild
    TYPE(NodeDefList), POINTER :: childlist, overlapchildlist
    INTEGER :: n

    childlist=>node%children 

    DO WHILE (associated(childlist))

       child=>childlist%self 

       IF (NephewCanBeOverlap(overlap,child,n)) THEN
          overlapchildlist=>overlap%children                    
          DO WHILE (associated(overlapchildlist))                                            
             overlapchild=>overlapchildlist%self 
             IF (Overlaps(child,overlapchild,n+1)) CALL AddOverlap(child,overlapchild)

             overlapchildlist=>overlapchildlist%next
          END DO
       END IF

       childlist=>childlist%next
    END DO

  END SUBROUTINE InheritOverlapChildren


  !> @brief Looks to see what old children of overlap should be overlaps of children
  !! @param node node whose children need overlaps
  !! @param overlap overlapping node 
  !! @param n level
  SUBROUTINE InheritOverlapOldChildren(node,overlap,n)
    TYPE(NodeDef), POINTER :: node, child, overlap, overlapchild
    TYPE(NodeDefList), POINTER :: childlist, overlapchildlist
    INTEGER :: n

    childlist=>node%children 
    DO WHILE (associated(childlist))
       child=>childlist%self 
       IF (NephewCanBeOverlap(overlap,child,n)) THEN
          overlapchildlist=>overlap%oldchildren 
          DO WHILE (associated(overlapchildlist))
             overlapchild=>overlapchildlist%self 
             IF (Overlaps(child,overlapchild,n+1)) CALL AddOverlap(child,overlapchild) 
             overlapchildlist=>overlapchildlist%next
          END DO
       END IF
       childlist=>childlist%next
    END DO

  END SUBROUTINE InheritOverlapOldChildren

  !> @brief Looks to see what children of overlap should be overlaps of old children
  !! @param node node whose old children need overlaps
  !! @param overlap overlapping node 
  !! @param n level
  SUBROUTINE InheritOverlapNewChildren(node,overlap,n)
    TYPE(NodeDef), POINTER :: node, child, overlap, overlapchild
    TYPE(NodeDefList), POINTER :: childlist, overlapchildlist
    INTEGER :: n
    childlist=>node%oldchildren 
    DO WHILE (associated(childlist))
       child=>childlist%self 
       IF (NephewCanBeOverlap(overlap,child,n)) THEN
          overlapchildlist=>overlap%children 
          DO WHILE (associated(overlapchildlist))
             overlapchild=>overlapchildlist%self 
             IF (Overlaps(child,overlapchild,n+1)) CALL AddOverlap(child,overlapchild) 
             overlapchildlist=>overlapchildlist%next
          END DO
       END IF
       childlist=>childlist%next
    END DO

  END SUBROUTINE InheritOverlapNewChildren

  !> @brief Looks to see what children of oneself should be neighbors of each other
  !! @param node node whose children need neighbors
  !! @param n level
  SUBROUTINE InheritSelfNeighborChildren(node,n)
    TYPE(NodeDef), POINTER :: node, child, sibling
    TYPE(NodeDefList), POINTER :: childlist, siblinglist
    INTEGER :: n
    childlist=>node%children 
    DO WHILE (associated(childlist))
       child=>childlist%self 
       siblinglist=>childlist%next 
       DO WHILE (associated(siblinglist))
          sibling=>siblinglist%self 
          IF (Neighbors(child,sibling,n+1)) CALL AddNeighbor(child,sibling) 
          IF (Neighbors(child,sibling,n+1)) CALL AddNeighbor(sibling,child) 
          siblinglist=>siblinglist%next
       END DO
       childlist=>childlist%next 
    END DO
  END SUBROUTINE InheritSelfNeighborChildren


  !> @brief Looks to see what children of a given node should be overlaps with nodes oldchildren
  !! @param node node whose children and oldchildren need overlaps
  !! @param n level
  SUBROUTINE InheritSelfOverlapOldChildren(node,n)
    TYPE(NodeDef), POINTER :: node, child, overlapchild
    TYPE(NodeDefList), POINTER :: childlist, overlapchildlist
    INTEGER :: n

    childlist=>node%children 
    DO WHILE (associated(childlist))
       child=>childlist%self 
       overlapchildlist=>node%oldchildren 
       DO WHILE (associated(overlapchildlist))
          overlapchild=>overlapchildlist%self          
          IF (Overlaps(child,overlapchild,n+1)) THEN
             CALL AddOverlap(child,overlapchild) 
          END IF
          overlapchildlist=>overlapchildlist%next            
       END DO
       childlist=>childlist%next
    END DO
  END SUBROUTINE InheritSelfOverlapOldChildren

  !> @brief Looks to see what old children of a given node should be overlaps with nodes children
  !! @param node node whose children and oldchildren need overlaps
  !! @param n level
  SUBROUTINE InheritSelfOverlapNewChildren(node,n)
    TYPE(NodeDef), POINTER :: node, child, overlapchild
    TYPE(NodeDefList), POINTER :: childlist, overlapchildlist
    INTEGER :: n

    childlist=>node%children 
    DO WHILE (associated(childlist))
       child=>childlist%self 
       overlapchildlist=>node%oldchildren 
       DO WHILE (associated(overlapchildlist))
          overlapchild=>overlapchildlist%self          
          IF (Overlaps(child,overlapchild,n+1)) THEN
             CALL AddOverlap(overlapchild,child) 
          END IF
          overlapchildlist=>overlapchildlist%next            
       END DO
       childlist=>childlist%next
    END DO
  END SUBROUTINE InheritSelfOverlapNewChildren
  !> @}


  !> @name Single node operations
  !! @{

  !> @brief Creates and distributes new children of node
  !! @param node node that is creating children
  !! @param n level
  SUBROUTINE CreateChildren(node,n)
    INTEGER :: n, step
    TYPE(NodeDef), POINTER :: node, child
    INTEGER, POINTER, DIMENSION(:,:,:) :: childgrids
    TYPE(NodeBox), POINTER :: child_box
    INTEGER :: iErr,j,i,k
    INTEGER :: nprocs = 0
    INTEGER :: nchildren
    step=levels(n)%step

    NULLIFY(child)
    NULLIFY(childgrids)
    NULLIFY(child_box)

    IF (n >= -2) THEN
       IF (lRegrid .AND. n == 0) THEN
          nchildren=1
          ALLOCATE(childgrids(3,2,nChildren))
          childgrids(:,:,1)=node%box%mGlobal-spread(node%box%mGlobal(:,1), 2, 2)+1
       ELSEIF (lStressTest) THEN
          IF (n == -2) THEN              
             nchildren=nDomains
             ALLOCATE(childgrids(3,2,nchildren))
             DO j=1,nChildren
                childgrids(:,:,j)=Domains(j)%mGlobal
             END DO
          ELSE IF (n == -1) THEN
!             nchildren=1
!             ALLOCATE(childgrids(3,2,1))
!             childgrids(:,:,1)=node%box%mGlobal
             IF (nDim == 2) THEN
                IF (MPI_NP<=2) THEN
                   nchildren=1
                   ALLOCATE(childgrids(3,2,nchildren))
                   childgrids=1
                   childgrids(1:2,:,1)=RESHAPE((/1,1,8,8/),(/2,2/))
!                   childgrids(1:2,:,2)=RESHAPE((/17,1,32,16/),(/2,2/))
                   !                     childgrids(1:2,:,1)=RESHAPE((/1,1,32,16/),(/2,2/))
                ELSE IF (MPI_NP==4) THEN
                   nchildren=4
                   ALLOCATE(childgrids(3,2,4))
                   childgrids=1
                   childgrids(1:2,:,1)=RESHAPE((/1,1,16,8/),(/2,2/))
                   childgrids(1:2,:,2)=RESHAPE((/17,1,32,8/),(/2,2/))
                   childgrids(1:2,:,3)=RESHAPE((/1,9,16,16/),(/2,2/))
                   childgrids(1:2,:,4)=RESHAPE((/17,9,32,16/),(/2,2/))
                END IF
             ELSEIF (nDim == 3) THEN
                IF (MPI_NP<=2) THEN
                   nchildren=2!1
                   ALLOCATE(childgrids(3,2,nchildren))
                   childgrids(1:3,:,1)=RESHAPE((/1,1,1,16,16,16/),(/3,2/))
                   childgrids(1:3,:,2)=RESHAPE((/17,1,1,32,16,16/),(/3,2/))
                   !                     childgrids(1:3,:,1)=RESHAPE((/1,1,1,32,16,16/),(/3,2/))
                ELSEIF (MPI_NP == 4) THEN
                   nchildren=4
                   ALLOCATE(childgrids(3,2,4))
                   childgrids(1:3,:,1)=RESHAPE((/1,1,1,16,8,16/),(/3,2/))
                   childgrids(1:3,:,2)=RESHAPE((/17,1,1,32,8,16/),(/3,2/))
                   childgrids(1:3,:,3)=RESHAPE((/1,9,1,16,16,16/),(/3,2/))
                   childgrids(1:3,:,4)=RESHAPE((/17,9,1,32,16,16/),(/3,2/))
                END IF
             END IF
          ELSEIF (n == 0) THEN
             nchildren=1
             ALLOCATE(childgrids(3,2,1))
             childgrids=1
             IF (nDim == 2) THEN
                IF (MPI_NP==4) THEN
                   IF (ALL(node%box%mGlobal(1:2,1)==(/1,1/))) THEN
                      childgrids(1:2,:,1)=RESHAPE((/9,5,16,8/),(/2,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:2,1)==(/1,9/))) THEN
                      childgrids(1:2,:,1)=RESHAPE((/9,1,16,4/),(/2,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:2,1)==(/17,1/))) THEN
                      childgrids(1:2,:,1)=RESHAPE((/1,5,8,8/),(/2,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:2,1)==(/17,9/))) THEN
                      childgrids(1:2,:,1)=RESHAPE((/1,1,8,4/),(/2,2/))
                   END IF
                ELSEIF (MPI_NP<=2) THEN
!                   write(*,*) MPI_ID, node%box%mGlobal
                   IF (LevelBalance(1)== 0) THEN
                      IF (ALL(node%box%mGlobal(1:2,1)==(/1,1/))) THEN
                         childgrids(1:2,:,1)=RESHAPE((/3,3,4,6/),(/2,2/))
                      ELSE
                         childgrids(1:2,:,1)=RESHAPE((/1,3,2,6/),(/2,2/))
                      END IF
                   ELSE
                      IF (ALL(node%box%mGlobal(1:2,1)==(/1,1/))) THEN
                         childgrids(1:2,:,1)=RESHAPE((/3,3,6,6/),(/2,2/))
                      ELSEIF (ALL(node%box%mGlobal(1:2,1)==(/17,1/))) THEN
                         childgrids(1:2,:,1)=RESHAPE((/5,7,9,10/),(/2,2/))
                      END IF
                   END IF
                END IF
             ELSEIF (nDim == 3) THEN
                IF (MPI_NP==4) THEN
                   IF (ALL(node%box%mGlobal(1:3,1)==(/1,1,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/9,5,1,16,8,1/),(/3,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:3,1)==(/1,9,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/9,1,1,16,4,1/),(/3,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:3,1)==(/17,1,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/1,5,1,8,8,1/),(/3,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:3,1)==(/17,9,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/1,1,1,8,4,1/),(/3,2/))
                   END IF
                ELSEIF (MPI_NP<=2) THEN
                   IF (ALL(node%box%mGlobal(1:3,1)==(/1,1,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/13,7,7,16,10,10/),(/3,2/))
                   ELSEIF (ALL(node%box%mGlobal(1:3,1)==(/17,1,1/))) THEN
                      childgrids(1:3,:,1)=RESHAPE((/5,7,7,9,10,10/),(/3,2/))
                   END IF
                END IF
             END IF
          ELSEIF (n == 1) THEN
             IF (MPI_NP<=2) THEN
                IF (ALL(node%box%mGlobal(1:2,1)==(/25,13/))) THEN
                   nchildren=1
                   ALLOCATE(childgrids(3,2,1))
                   childgrids=1
                   childgrids(1:2,:,1)=RESHAPE((/1,1,4,4/),(/2,2/))
                ELSE
                   nchildren=0
                END IF
             END IF
          ELSE
             CALL NewSubGrids(node%info, nchildren, childgrids)
          END IF
       ELSE
          IF (n == -2) THEN              
             nchildren=nDomains
             ALLOCATE(childgrids(3,2,nchildren))
             DO j=1,nChildren
                childgrids(:,:,j)=Domains(j)%mGlobal
             END DO
          ELSE IF (n == -1) THEN
             !               IF (ASSOCIATED(node%proclist)) THEN
             !                  nprocs = SIZE(node%proclist)
             !               END IF
!             nchildren=2
!             ALLOCATE(childgrids(3,2,nchildren))
!             childgrids(:,:,1)=reshape((/1,1,1,16,16,8/),(/3,2/))
!             childgrids(:,:,2)=reshape((/1,1,9,16,16,16/),(/3,2/))
 
            nchildren=1
             ALLOCATE(childgrids(3,2,nchildren))
             childgrids(:,:,1)=node%box%mGlobal
          ELSE  
             CALL NewSubGrids(node%info, nchildren, childgrids)                        
          END IF
       END IF

       IF (nChildren > 0) THEN
          CALL CreateNodeBox(node%box%mGlobal, child_box, MPI_ID)
          DO j=1, nChildren             
             childgrids(:,:,j)=levelUp(childgrids(:,:,j)+spread(node%box%mGlobal(:,1)-1,2,2), n)
          END DO
          CALL HilbertSort(childgrids, n+1)
          DO j=1, nChildren             
             child_box%mGlobal=childgrids(:,:,j)
             NULLIFY(child)
             IF (MPI_NP == 1) THEN
                CALL AddNode(n+1, child_box, child)
                CALL AddParent(child, node)
             ELSE
                ALLOCATE(child)
                CALL NullifyNodeFields(child)
                child%box=child_box
             END IF
             CALL AddChild(node, child)
          END DO
       END IF
       IF (MPI_NP == 1 .AND. n >= 0) CALL AllocChildFixups(node%info, childgrids)
       IF (nChildren > 0) THEN
          CALL DestroyNodeBox(child_box)
          DEALLOCATE(childgrids, STAT=iErr)
          IF (iErr /= 0) THEN
             PRINT *, "CreateChildren() error:  Unable to deallocate childgrids."
             STOP
          END IF
          NULLIFY(childgrids)
       END IF
    END IF

  END SUBROUTINE CreateChildren


  !> Move a node's children to the oldchildren list.
  !! @param node node object
  SUBROUTINE AgeNodeChildren(node)

    TYPE(NodeDef) :: node

    ! Move node's children to its old children list and then clear
    ! the children list.
    CALL ClearNodeList(node%oldchildren)
    node%oldchildren=>node%children
    NULLIFY(node%children)
    NULLIFY(node%lastchild)

  END SUBROUTINE AgeNodeChildren



   !> Routine that restores child relationships from backup nodes
   !! @param n level
   !! @param node node object needing parent child relationships

  SUBROUTINE RestoreChildren(n, node, lstrict_opt)
      TYPE(NodeDef), POINTER :: node, parent, currentnode, child
      TYPE(NodeDefList), POINTER :: children
      INTEGER :: n
      LOGICAL, OPTIONAL :: lstrict_opt
      LOGICAL :: lstrict
      IF (PRESENT(lstrict_opt)) THEN
         lstrict=lstrict_opt
      ELSE
         lstrict=.true.
      END IF
      NULLIFY(currentnode)
      IF (lstrict) THEN
         CALL StrictFindBackupNode(n,node%box, currentnode, "RestoreChildren")
      ELSE
         CALL FindBackupNode(n, node%box, currentnode)
      END IF
      IF (ASSOCIATED(currentnode)) THEN
         children=>currentnode%children
         DO WHILE (ASSOCIATED(children))
            CALL StrictFindNode(n+1,children%self%box, child, "RestoreChildren")
            CALL AddChild(node, child)
            children=>children%next
         END DO
      END IF
   END SUBROUTINE RestoreChildren


   !> Routine that restores child relationships from backup nodes
   !! @param n level
   !! @param node node object needing parent child relationships

  SUBROUTINE RestoreParent(n, node)
      TYPE(NodeDef), POINTER :: node, parent, currentnode
      INTEGER :: n     

      CALL StrictFindBackupNode(n,node%box, currentnode, "RestoreParent")
      CALL StrictFindNode(n-1,currentnode%parent%box, parent, "RestoreParent")
      CALL AddParent(node, parent)
   END SUBROUTINE RestoreParent

   !> Routine that backs up child relationships from nodes
   !! @param n level
   !! @param node node object needing parent child relationships

  SUBROUTINE BackupChildren(n, node)
      TYPE(NodeDef), POINTER :: node, parent, currentnode, child
      TYPE(NodeDefList), POINTER :: children
      INTEGER :: n
      CALL StrictFindNode(n,node%box, currentnode, "BackupChildren")
      children=>currentnode%children
      DO WHILE (ASSOCIATED(children))
         CALL StrictFindBackupNode(n+1,children%self%box, child, "BackupChildren")
         CALL AddChild(node, child)
         children=>children%next
      END DO
   END SUBROUTINE BackupChildren


   !> Routine that restores child relationships from backup nodes
   !! @param n level
   !! @param node node object needing parent child relationships

  SUBROUTINE BackupParent(n, node)
      TYPE(NodeDef), POINTER :: node, parent, currentnode
      INTEGER :: n     
      CALL StrictFindNode(n,node%box, currentnode, "BackupParent")
      CALL StrictFindBackupNode(n-1,currentnode%parent%box, parent, "BackupParent")
      CALL AddParent(node, parent)
   END SUBROUTINE BackupParent



  !> @}

END MODULE TreeNodeOps
