!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    tree_parsing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file tree_parsing.f90
!! @brief Main file for module TreeParsing

!> @defgroup TreeParsing Tree Parsing
!! @brief Performs necessary communication required for pair-wise node operations 
!! and calls appropriate pair-wise node routine when unparsing messages.
!! Intermediate between TreeLevelComms and TreeNodeOps
!! @ingroup TreeComms

!> @brief Performs necessary communication of new tree tree created by CreateChildren 
!! to necessary processors and calls appropriate pair-wise node routine when unparsing messages.
!! Intermediate between TreeLevelComms and TreeNodeOps
!! @ingroup TreeParsing
MODULE TreeParsing
   USE MPIPacking
   USE MessageDeclarations
   USE TreeNodeOps

   IMPLICIT NONE
   PUBLIC
CONTAINS

   !> @name NeighboringChildrenParsing Routines for parsing nodes for InheritNeighborChildren
   !! @ingroup TreeNodeParsing
   !! @{ 


   !> @brief sends children that might neighbor neighbor's children
   !! @param message message to pack to
   !! @param node The node whose children should be packed
   !! @param neighbor The neighbor node that might have children that neighbor the node's children
   SUBROUTINE SendNeighboringChildren(message,node,neighbor)
      TYPE(NodeDef), POINTER :: node, neighbor, child
      TYPE(NodeDefList), POINTER :: childlist
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nChildren,level

      childlist=>node%children 
      nChildren=0 !Iterate through once to get a count
      level=message%level
      DO WHILE (associated(childlist))
         child=>childlist%self 
         IF (NephewCanBeOverlap(neighbor,child,level))  nChildren=nChildren+1
         childlist=>childlist%next         
      END DO
      IF (nChildren > 0) THEN
         CALL PackData(message, neighbor%box%mGlobal) !So the neighbor knows the parent node
         CALL PackData(message, node%box%mGlobal) !So the neighbor knows the parent node
         CALL PackData(message,nChildren)
         childlist=>node%children 
         DO WHILE (associated(childlist))
            child=>childlist%self 
            IF (NephewCanBeOverlap(neighbor,child,level)) CALL PackData(message,child%box)
            childlist=>childlist%next         
         END DO
      END IF
   END SUBROUTINE SendNeighboringChildren


   !> @brief receives children that might neighbor children of local nodes
   !! @param message message to unpack from
   SUBROUTINE RecvNeighboringChildren(message)
      TYPE(NodeDef), POINTER :: node, neighbor, child
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nChildren,i
      INTEGER :: level
      TYPE(NodeBox) :: neighborbox, childbox, localnodebox


      level=message%level
      neighborbox%MPI_ID = message%remote_proc
      localnodebox%MPI_ID=MPI_ID
      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvNeighboringChildren"))

         NULLIFY(node, neighbor)

         CALL StrictFindNode(level,localnodebox, node, "RecvNeighboringChildren(node)")
         CALL UnpackData(message,neighborbox%mGlobal)

         CALL StrictFindNode(level,neighborbox,neighbor, "RecvNeighboringChildren(neighbor)")
         CALL UnpackData(message,nChildren)

         DO i=1,nChildren
            NULLIFY(child)

            CALL UnPackData(message,childbox)

            CALL AddFindNode(level+1,childbox,child)
            CALL AddFindChild(neighbor,child)
         END DO
         CALL InheritNeighborChildren(node,neighbor,level)
      END DO

   END SUBROUTINE RecvNeighboringChildren


   !> @}

   !> @name OverlappingChildrenParsing Routines for parsing nodes for InheritOverlapChildren
   !! @{ 

   !> @brief This routine is called by two different schedulers:
   !! PostSendOverlappingChildrenToNewNodes
   !! PostSendOverlappingChildrenToOldNodes
   !! Both of which send children to overlaps
   !! (For old grids, the children info is being sent to new overlaps)
   !! (For new grids, the children info is being sent to old overlaps)
   !! @param message The message to pack to
   !! @param node The node whose children should be packed
   !! @param overlap The overlap node that might have children that overlap the node's children
   SUBROUTINE SendOverlappingChildren(message,node,overlap)
      TYPE(NodeDef), POINTER :: node, overlap, child
      TYPE(NodeDefList), POINTER :: childlist
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nChildren,level

      childlist=>node%children 
      nChildren=0 !Iterate through once to get a count
      level=message%level
      DO WHILE (associated(childlist))
         child=>childlist%self 
         IF (NephewCanBeOverlap(overlap,child,level))  nChildren=nChildren+1
         childlist=>childlist%next         
      END DO
      IF (nChildren > 0) THEN
         CALL PackData(message, overlap%box%mGlobal) !So the neighbor knows the parent node
         CALL PackData(message, node%box%mGlobal) !So the neighbor knows the parent node
         CALL PackData(message,nChildren)
         childlist=>node%children 
         DO WHILE (associated(childlist))
            child=>childlist%self 
            IF (NephewCanBeOverlap(overlap,child,level)) CALL PackData(message,child%box)
            childlist=>childlist%next         
         END DO
      END IF
   END SUBROUTINE SendOverlappingChildren

   !> @brief receives children that might overlap children of local nodes
   !! @param message message to unpack from
   SUBROUTINE RecvOverlappingChildrenFromOldNodes(message)
      TYPE(NodeDef), POINTER :: node, overlap, child
      TYPE(NodeDefList), POINTER :: overlapchildlist
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nChildren,i
      INTEGER :: level
      TYPE(NodeBox) :: localnodebox, overlapbox, childbox


      level=message%level
      localnodebox%MPI_ID = MPI_ID
      NULLIFY(overlap)


      overlapbox%MPI_ID=message%remote_proc

      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvOverlappingChildrenFromOldNodes"))

         NULLIFY(node)

         CALL StrictFindNode(level,localnodebox,node, "RecvOverlappingChildrenFromNodes(node)")
         CALL UnPackData(message, overlapbox%mGlobal)
         CALL StrictFindOldNode(level,overlapbox,overlap, "RecvOverlappingChildrenFromNodes(overlap)")
         CALL UnpackData(message, nChildren)

         IF (nChildren > 0) THEN
   
             DO i=1,nChildren
                 ! Retrieve the new child box from the packed message.
                 CALL UnpackData(message,childbox)
                 ! Find a child node on the next level up, or add one if none exists.
                 NULLIFY(child)
                 CALL AddFindOldNode(level+1, childbox, child)
                 ! Search for this child node within overlap's list, and add it if it isn't found.
                 CALL AddFindChild(overlap, child)
             END DO

             CALL InheritOverlapChildren(node,overlap,level)
         END IF
      END DO

   END SUBROUTINE RecvOverlappingChildrenFromOldNodes

   !> @brief receives children that might overlap children of old nodes
   !! @param message message to unpack from
   SUBROUTINE RecvOverlappingChildrenFromNewNodes(message)
      TYPE(NodeDef), POINTER :: node, overlap, child
      TYPE(NodeDefList), POINTER :: overlapchildlist
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nChildren,i
      INTEGER :: level
      TYPE(NodeBox) :: localnodebox, overlapbox, childbox


      level=message%level
      localnodebox%MPI_ID = MPI_ID

      NULLIFY(overlap)

      overlapbox%MPI_ID=message%remote_proc

      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvOverlappingChildrenFromNewNodes(local)"))

         NULLIFY(node)
         NULLIFY(overlap)

         CALL StrictFindOldNode(level,localnodebox,node, "RecvOverlappingChildrenFromNewNodes(node)")
         CALL UnPackData(message, overlapbox%mGlobal)
         CALL StrictFindNode(level,overlapbox,overlap, "RecvOverlappingChildrenFromNewNodes(overlap)")
         CALL UnpackData(message, nChildren)

         IF (nChildren > 0) THEN

             DO i=1,nChildren
                 CALL UnpackData(message,childbox)
                 CALL AddFindNode(level+1, childbox, child)
                 ! Search for this child node within overlap's list, and add it
                 ! if it isn't found.
                 CALL AddFindChild(overlap, child)
             END DO

         END IF

         CALL InheritOverlapChildren(node,overlap,level)
      END DO

   END SUBROUTINE RecvOverlappingChildrenFromNewNodes

   !> @}

   !> @name ChildGridParsing Routines for parsing new nodes to child processors
   !! @{

   !> @brief Informs children of their new grids or of their nextlevel parentprocs
   !! @param message The message to pack to
   !! @param child The child node to pack
   SUBROUTINE SendGridToChild(message, child)
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeDef), POINTER :: child

      CALL PackData(Message, child%box%mGlobal)
!      CALL PackList(Message, child%proclist)
!      CALL PackList(Message, child%proctime)
      CALL PackData(Message, child%parent%box%mGlobal)      

   END SUBROUTINE SendGridToChild

   !> @brief Receiving incoming new grids 
   !! @param message The message to unpack from
   SUBROUTINE RecvGridsFromParent(message)
      TYPE(PackedMessage), POINTER :: message
      Type(NodeDef), POINTER:: node, parent
      TYPE(NodeBox) :: box, parentbox
      INTEGER :: level


      box%MPI_ID=MPI_ID
      parentbox%MPI_ID=message%remote_proc

      DO WHILE (StrictGetNextBox(message,box%mGlobal, "RecvGridsFromParent"))
         
         NULLIFY(node, parent)
         CALL AddFindNode(message%level+1,box,node)
         CALL UnPackData(message,parentbox%mGlobal)

         CALL AddFindNode(message%level,parentbox,parent)
         CALL AddParent(node,parent)
      END DO

   END SUBROUTINE RecvGridsFromParent

   !> @}

   !> @name ChildOverlapsNeighborsParsing Routines for parsing overlaps and neighbors to new children
   !! @{ 
  
   !> @brief Informs children of their neighbors and overlaps
   !! @param message The message to pack to
   !! @param child The child node to pack neighbors and overlaps of
   SUBROUTINE SendOverlapsNeighborsToChild(Message, child)
      TYPE(NodeDef), POINTER :: child
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: nOverlaps, nNeighbors
      TYPE(NodeDefList), POINTER :: nodelist


      CALL PackData(message, child%box%mGlobal) !Pack the child box before its overlaps and neighbors
      nOverlaps=NodeCount(child%overlaps)

      CALL PackData(message, nOverlaps)

      IF (nOverlaps > 0) THEN
         nodelist=>child%overlaps
         DO WHILE(ASSOCIATED(nodelist))

            CALL PackData(Message,nodelist%self%box)
            nodelist=>nodelist%next
         END DO
      END IF

      nNeighbors=NodeCount(child%Neighbors)

      CALL PackData(message, nNeighbors)

      IF (nNeighbors > 0) THEN
         nodelist=>child%neighbors
         DO WHILE(ASSOCIATED(nodelist))
            CALL PackData(Message,nodelist%self%box)
            nodelist=>nodelist%next
         END DO
      END IF

   END SUBROUTINE SendOverlapsNeighborsToChild

   !> @brief Receives neighbors and overlaps of new grids received by RecvGridsFromParent
   !! @param message The message to unpack from
   SUBROUTINE RecvOverlapsNeighbors(message)
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeBox) :: node_box, overlapbox, neighborbox
      TYPE(NodeDef), POINTER :: node, overlap, neighbor
      INTEGER :: level, nOverlaps, nNeighbors, i

      level=message%level+1
      node_box%MPI_ID = MPI_ID

      DO WHILE(StrictGetNextBox(message, node_box%mGlobal, "RecvOverlapsNeighbors"))

         NULLIFY(node)
         CALL StrictFindNode(level,node_box,node, "RecvOverlapsNeighbors(node)")
         CALL UnpackData(message, nOverlaps)

         DO i=1,nOverlaps
            NULLIFY(overlap)
            CALL UnPackData(message, overlapbox)
            CALL AddFindOldNode(level,overlapbox,overlap)
            CALL AddOverlap(node, overlap)
         END DO

         CALL UnpackData(message, nNeighbors)

         DO i=1,nNeighbors
            NULLIFY(neighbor)
            CALL UnPackData(message, neighborbox)
            CALL AddFindNode(level,neighborbox,neighbor)
            CALL AddNeighbor(node, neighbor)
         END DO

      END DO

   END SUBROUTINE RecvOverlapsNeighbors

   !> @}

   !> @name OldChildOverlapsParsing Routines for parsing overlaps to old children
   !! @{ 

   !> @brief This routine is called by two different schedulers:
   !! PostSendOverlapsToOldNodesChildren
   !! PostSendOverlapsToNodesOldChildren
   !! Both of which send new overlaps to old grids on the child level
   !! (on step 1 the old child grids are old grids children)
   !! (on step 2 the old child grids are the current grids old children)
   !! @param message The message to pack to
   !! @param child The child node to pack the overlaps of
   SUBROUTINE SendOverlapsToChild(message, child)
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeDef), POINTER :: child

      INTEGER :: nOverlaps
      TYPE(NodeDefList), POINTER :: nodelist


      CALL PackData(message, child%box%mGlobal) !Pack the child box before its overlaps and neighbors
      nOverlaps=NodeCount(child%overlaps)
      CALL PackData(message, nOverlaps)

      nodelist=>child%overlaps
      DO WHILE(ASSOCIATED(nodelist))
         CALL PackData(Message,nodelist%self%box)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE SendOverlapsToChild


   !> @brief This routine is called by old grids and unpacks new overlaps received from the parent grids
   !! @param message The message to unpack from
   SUBROUTINE RecvOldNodeOverlaps(message)
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeBox) :: node_box, overlapbox
      TYPE(NodeDef), POINTER :: node, overlap
      INTEGER :: level,i,nOverlaps
      level=message%level+1
      node_box%MPI_ID = MPI_ID


      DO WHILE(StrictGetNextBox(message, node_box%mGlobal, "RecvOldNodeOverlaps"))
         NULLIFY(node)
         CALL StrictFindOldNode(level,node_box,node, "RecvOldNodeOverlaps")
         CALL UnpackData(message, nOverlaps)

         DO i=1,nOverlaps

            NULLIFY(overlap)

            CALL UnPackData(message, overlapbox)
            CALL AddFindNode(level,overlapbox,overlap)
            CALL AddOverlap(node, overlap)
         END DO
      END DO
   END SUBROUTINE RecvOldNodeOverlaps

   !> @}

   !> @name ControlParsingRoutines Routines for parsing parent processors
   !! @{ 


   !> @brief This routine extracts the list of new parent procs for the grandchildren of child grids
   !! @param MessageGroup The message group containing the child grids
   !! @param ChildChildArray A 2D array of size MPI_NPxMPI_NP where ChildChildArray(i,j) is true 
   !! @param GrandChildArray 1D array containing child and grandchild procs
   !! if processor i is on a proclist of any child assigned to processor j
   SUBROUTINE FinalizeSendGridsToChildren(MessageGroup, ChildChildArray, GrandChildArray)
      TYPE(StageMessageGroup), POINTER :: MessageGroup
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:), POINTER :: NewParentList
      INTEGER, DIMENSION(:), POINTER :: GrandChildProcList
      INTEGER :: i
      LOGICAL, DIMENSION(:,:), POINTER :: ChildChildArray
      LOGICAL, DIMENSION(:), POINTER :: GrandChildArray
      

      NULLIFY(NewParentList, GrandChildProcList)

      CALL FindList1D(GrandChildArray, GrandChildProcList)

      DO i=1,size(GrandChildProcList)

         IF (GrandChildProcList(i) /= MPI_id) THEN
            CALL ExtractMessageFromGroup(MessageGroup, GrandChildProcList(i), message)
            CALL PackData(message, TERMINATIONBOX)
            IF (message%level < MaxLevel-1) THEN 
               CALL FindList2D(ChildChildArray,GrandChildProcList(i),NewParentList)
               CALL PackList(message, NewParentList)
               IF (ASSOCIATED(NewParentList)) THEN
                  DEALLOCATE(NewParentList)
                  NULLIFY(NewParentList)
               END IF
            END IF
         END IF

      END DO

      IF (ASSOCIATED(GrandChildProcList)) THEN
         DEALLOCATE(GrandChildProcList)
         NULLIFY(GrandChildProcList)
      END IF

   END SUBROUTINE FinalizeSendGridsToChildren

   !> @brief This routine receives the parentlist for the next level from the parents of the current level
   !! @param MessageGroup The message group containing the child grids
   !! @param NewParentsArray A 1D array of size MPI_NP where NewParentsArray(i) is true 
   !! if the proclists of any grid on processor i contains the current processor.
   SUBROUTINE FinalizeRecvGridsFromParents(MessageGroup, NewParentsArray)
      TYPE(StageMessageGroup), POINTER :: MessageGroup
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:), POINTER :: ParentProcList, NewParentList
      INTEGER :: i, n
      INTEGER, DIMENSION(1) :: lb, ub
      LOGICAL, DIMENSION(:), POINTER :: NewParentsArray


      NULLIFY(NewParentList, ParentProcList)

      CALL GetProcListAsArray(MessageGroup, ParentProcList)

      IF (ASSOCIATED(ParentProcList)) THEN
         DO i=1,size(ParentProcList)

            CALL ExtractMessageFromGroup(MessageGroup, ParentProcList(i), message)
            CALL UnpackList(message, NewParentList)

            lb = LBOUND(NewParentList)
            ub = UBOUND(NewParentList)

            DO n = lb(1), ub(1)
               NewParentsArray(NewParentList(n))=.TRUE.
            END DO

            IF (ASSOCIATED(NewParentList)) THEN
               DEALLOCATE(NewParentList)
               NULLIFY(NewParentList)
            END IF
         END DO

         IF (ASSOCIATED(ParentProcList)) THEN
            DEALLOCATE(ParentProcList)
            NULLIFY(ParentProcList)
         END IF

      END IF

   END SUBROUTINE FinalizeRecvGridsFromParents

   !> @}

   
   !> @brief Returns a 1D array 'b' containing the true indices from column j of 2D logical array 'A'
   !! @param A 2D logical array whose lower bounds are expected to be 0
   !! @param j Column index
   !! @param b Output array that needs to be allocated and returned
   SUBROUTINE FindList2D(A,j,b)
      LOGICAL, DIMENSION(:,:), POINTER :: A
      INTEGER :: i,j
      INTEGER, DIMENSION(:), POINTER :: b
      INTEGER :: n
      n=COUNT(A(:,j))
      ALLOCATE(b(n))
      n=0
      DO i=0,size(A,1)-1
         IF (A(i,j)) THEN
            n=n+1
            b(n)=i
         END IF
      END DO
   END SUBROUTINE FindList2D

   !> @brief Returns a 1D array 'b' containing the true indices of a 1D logical array 'A'
   !! @param A 1D logical array whose lower bound is expected to be 0
   !! @param b Output array that needs to be allocated and returned
   SUBROUTINE FindList1D(A,b)
      LOGICAL, DIMENSION(:), POINTER :: A
      INTEGER :: i
      INTEGER, DIMENSION(:), POINTER :: b
      INTEGER :: n

      NULLIFY(b)

      IF (ASSOCIATED(A)) THEN
         n=COUNT(A(:))
         ALLOCATE(b(n))
         n=0
         DO i=0,size(A,1)-1
            IF (A(i)) THEN
               n=n+1
               b(n)=i
            END IF
         END DO

      END IF

   END SUBROUTINE FindList1D

END MODULE TreeParsing

